package cn.com.duibaboot.ext.autoconfigure.flowreplay.serializer;

import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.Registration;
import com.esotericsoftware.kryo.factories.ReflectionSerializerFactory;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import com.esotericsoftware.kryo.serializers.CompatibleFieldSerializer;
import lombok.extern.slf4j.Slf4j;
import org.objenesis.strategy.StdInstantiatorStrategy;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.ConcurrentSkipListSet;

/**
 * kryo序列化工具，仅供录制回归使用
 * Created by guoyanfei .
 * 2019-04-18 .
 */
@Slf4j
public class KryoSerializer {

    private static final int BUFFER_SIZE = 4096 * 4;

    private static final Class<?> ARRAY_AS_LIST;
    private static final Class<?> UNMODIFIABLE_COLLECTION;
    private static final Class<?> UNMODIFIABLE_LIST;
    private static final Class<?> UNMODIFIABLE_MAP;
    private static final Class<?> UNMODIFIABLE_NAVIGABLEMAP;
    private static final Class<?> UNMODIFIABLE_NAVIGABLESET;
    private static final Class<?> UNMODIFIABLE_RANDOMACCESSLIST;
    private static final Class<?> UNMODIFIABLE_SET;
    private static final Class<?> UNMODIFIABLE_SORTEDMAP;
    private static final Class<?> UNMODIFIABLE_SORTEDSET;

    static {
        try {
            ARRAY_AS_LIST = Class.forName("java.util.Arrays$ArrayList");
            UNMODIFIABLE_COLLECTION = Class.forName("java.util.Collections$UnmodifiableCollection");
            UNMODIFIABLE_LIST = Class.forName("java.util.Collections$UnmodifiableList");
            UNMODIFIABLE_MAP = Class.forName("java.util.Collections$UnmodifiableMap");
            UNMODIFIABLE_NAVIGABLEMAP = Class.forName("java.util.Collections$UnmodifiableNavigableMap");
            UNMODIFIABLE_NAVIGABLESET = Class.forName("java.util.Collections$UnmodifiableNavigableSet");
            UNMODIFIABLE_RANDOMACCESSLIST = Class.forName("java.util.Collections$UnmodifiableRandomAccessList");
            UNMODIFIABLE_SET = Class.forName("java.util.Collections$UnmodifiableSet");
            UNMODIFIABLE_SORTEDMAP = Class.forName("java.util.Collections$UnmodifiableSortedMap");
            UNMODIFIABLE_SORTEDSET = Class.forName("java.util.Collections$UnmodifiableSortedSet");
        }
        catch (ClassNotFoundException e) {
            throw new IllegalStateException("should not be here", e);
        }
    }

    private static ThreadLocal<Kryo> kryoThreadLocal = ThreadLocal.withInitial(() -> {
        Kryo kryo = new Kryo(){
            public Registration writeClass (Output output, Class type) {
                if(type == ARRAY_AS_LIST){//java.util.Arrays$ArrayList用KRYO的默认实现在反序列化时会报错，这里用ArrayList代替
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_COLLECTION) {
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_LIST) {
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_MAP) {
                    type = HashMap.class;
                } else if (type == UNMODIFIABLE_NAVIGABLEMAP) {
                    type = ConcurrentSkipListMap.class;
                } else if (type == UNMODIFIABLE_NAVIGABLESET) {
                    type = ConcurrentSkipListSet.class;
                } else if (type == UNMODIFIABLE_RANDOMACCESSLIST) {
                    type = ArrayList.class;
                } else if (type == UNMODIFIABLE_SET) {
                    type = HashSet.class;
                } else if (type == UNMODIFIABLE_SORTEDMAP) {
                    type = ConcurrentSkipListMap.class;
                } else if (type == UNMODIFIABLE_SORTEDSET) {
                    type = ConcurrentSkipListSet.class;
                }
                return super.writeClass(output, type);
            }
        };

        //支持增删字段
        kryo.setDefaultSerializer(new ReflectionSerializerFactory(CompatibleFieldSerializer.class));
//		kryo.getFieldSerializerConfig().setFixedFieldTypes(true);
//		kryo.getFieldSerializerConfig().setOptimizedGenerics(true);
        kryo.setInstantiatorStrategy(new Kryo.DefaultInstantiatorStrategy(new StdInstantiatorStrategy()));
        return kryo;
    });

    private KryoSerializer() {
    }

    /**
     * 反序列化
     * @param bytes
     * @return
     */
    public static Object deserialize(byte[] bytes) {
        if (bytes == null || bytes.length == 0) {
            return null;
        }
        Kryo kryo = kryoThreadLocal.get();
        ByteArrayInputStream in = new ByteArrayInputStream(bytes);
        Input input = new Input(in, BUFFER_SIZE);
        Object obj = kryo.readClassAndObject(input);
        input.close();
        return obj;
    }

    /**
     * 序列化
     * @param object
     * @return
     */
    public static byte[] serialize(Object object) {
        if (object == null) {
            return null;
        }
        Kryo kryo = kryoThreadLocal.get();
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        Output output = new Output(out, BUFFER_SIZE);
        kryo.writeClassAndObject(output, object);
        output.close();
        return out.toByteArray();
    }

}
