package cn.com.duibaboot.ext.autoconfigure.grouping;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.grouping.filter.ServiceGroupFilter;
import cn.com.duibaboot.ext.autoconfigure.grouping.httpclient.ServiceGroupHttpAsyncClientPostProcessor;
import com.alibaba.ttl.TransmittableThreadLocal;
import com.netflix.discovery.EurekaClient;
import com.netflix.hystrix.HystrixCommand;
import feign.RequestInterceptor;
import feign.hystrix.HystrixFeign;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.rocketmq.client.consumer.listener.MessageListener;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.core.Ordered;

import javax.servlet.DispatcherType;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;

/**
 * 多场景测试的支持
 * Created by guoyanfei .
 * 2018/11/7 .
 */
@Configuration
@ConditionalOnClass({ TransmittableThreadLocal.class })
@EnableAspectJAutoProxy(proxyTargetClass = true)
public class ServiceGroupAutoConfiguration {

    /**
     * 配置Filter，在处理请求之前先标识服务分组标记
     */
    @Configuration
    @ConditionalOnClass({ TransmittableThreadLocal.class })
    @ConditionalOnWebApplication
    static class ServiceGroupFilterConfiguration {

        @Bean
        public ServiceGroupFilter serviceGroupFilter() {
            return new ServiceGroupFilter();
        }

        @Bean
        public FilterRegistrationBean serviceGroupFilterConfigurer(ServiceGroupFilter serviceGroupFilter) {
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(serviceGroupFilter);
            List<String> urlPatterns = new ArrayList<>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 4);
            return registrationBean;
        }

    }

    /**
     * 配置Feign请求拦截器，在请求前配置http头表示这是服务分组的请求
     */
    @Configuration
    @ConditionalOnClass({ HystrixCommand.class, HystrixFeign.class })
    public class ServiceGroupFeignConfiguration {

        @Bean
        public RequestInterceptor serviceGroupFeignRequestInterceptor() {
            return template -> {
                String groupKey = ServiceGroupContext.getCurrentGroupKey();
                if (StringUtils.isNotBlank(groupKey)) {
                    template.header(ServiceGroupUtils.DUIBA_SERVICE_GROUP_KEY, groupKey);//设置http请求头，传递给服务端，表示这个请求转发的服务需要调用分组的服务
                }
            };
        }

    }

    /**
     * 让CloseableHttpAsyncClient支持服务分组功能
     */
    @Configuration
    @ConditionalOnClass(CloseableHttpAsyncClient.class)
    public static class ServiceGroupHttpAsyncClientPostProcessorConfiguration{

        @Bean
        public static SpecifiedBeanPostProcessor serviceGroupHttpAsyncClientPostProcessor(){
            return new ServiceGroupHttpAsyncClientPostProcessor();
        }
    }

    /**
     * 加入AOP，多场景测试支持rocketmq, 如果当前服务器的场景ID刚好和消息中的场景ID相同，则直接处理即可；
     * 如果当前服务器的场景ID（或者当前服务器没有场景ID）和消息中的场景ID不同，则去eureka中找到具有相同场景ID的服务，并转发给对应服务处理。
     * @return
     */
    @Configuration
    @ConditionalOnClass({DefaultMQProducer.class, EurekaClient.class})
    @ConditionalOnBean(EurekaClient.class)
    static class RocketMqServiceGroupConfiguration{

        @Bean
        public RocketMqProducerServiceGroupAspect rocketMqProducerServiceGroupAspect(){
            return new RocketMqProducerServiceGroupAspect();
        }

        @Bean
        public static SpecifiedBeanPostProcessor<MessageListener> rocketMqMessageListenerPostProcessor4Group(){
            return new RocketMqMessageListenerPostProcessor4Group();
        }
    }

}
