package cn.com.duibaboot.ext.autoconfigure.perftest;

import cn.com.duiba.boot.perftest.InternalPerfTestContext;
import cn.com.duiba.boot.perftest.PerfTestUtils;
import cn.com.duiba.wolf.redis.RedisClient;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataRegister;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.etcd.client.EtcdKVClientDelegate;
import cn.com.duibaboot.ext.autoconfigure.perftest.filter.PerfTestFilter;
import cn.com.duibaboot.ext.autoconfigure.perftest.filter.PerfTestRejectProperties;
import cn.com.duibaboot.ext.autoconfigure.perftest.httpclient.HttpAsyncClientPostProcessor;
import com.alibaba.ttl.TransmittableThreadLocal;
import com.aliyun.openservices.ons.api.Consumer;
import com.mongodb.MongoClientURI;
import com.netflix.hystrix.HystrixCommand;
import feign.RequestInterceptor;
import feign.hystrix.HystrixFeign;
import io.shardingjdbc.core.jdbc.adapter.AbstractDataSourceAdapter;
import io.shardingjdbc.core.jdbc.core.ShardingContext;
import io.shardingjdbc.core.jdbc.core.datasource.MasterSlaveDataSource;
import io.shardingjdbc.core.jdbc.core.datasource.ShardingDataSource;
import io.shardingjdbc.core.rule.MasterSlaveRule;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.impl.nio.client.CloseableHttpAsyncClient;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.aspectj.lang.annotation.Aspect;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnResource;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.mongo.MongoProperties;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.EnableAspectJAutoProxy;
import org.springframework.context.annotation.Import;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.env.Environment;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.util.ReflectionUtils;
import redis.clients.jedis.Jedis;

import javax.annotation.Resource;
import javax.servlet.DispatcherType;
import javax.sql.DataSource;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;

/**
 * 线上压力测试自动配置
 *
 * Created by wenqi.huang on 2016/11/24.
 */
@Configuration
@ConditionalOnClass({TransmittableThreadLocal.class})
@EnableAspectJAutoProxy(proxyTargetClass = true)
@Import({WebSocketPerfTestConfiguration.class, PerfTestRejectProperties.class})
public class PerfTestAutoConfiguration {

    @Configuration
    //下面这行判断perftest.properties是否存在的逻辑必须存在，RocketMqPerfAspect中会用是否存在perfTestFootMarker来判断是否接入了压测
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnBean(EtcdKVClientDelegate.class)
    static class PerfTestFootMarkerConfiguration {

        @Bean
        public PerfTestFootMarker perfTestFootMarker(EtcdKVClientDelegate etcdKVClientDelegate) {
            return new PerfTestFootMarker(etcdKVClientDelegate);
        }

    }

    /**
     * 嵌入 Filter，处理包含_duibaPerf=1参数的url(或cookie中有_duibaPerf=1)，数据库全部走影子库
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({TransmittableThreadLocal.class})
    @ConditionalOnWebApplication
    static class PerfTestFilterConfiguration {

        @Bean
        public PerfTestFilter perfTestFilter(){
            return new PerfTestFilter();
        }

        @Bean
        public FilterRegistrationBean perfTestFilterConfigurer(PerfTestFilter perfTestFilter){
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(perfTestFilter);
            List<String> urlPatterns=new ArrayList<>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 3);//这个Filter要在最前面,但是一定要在OrderedCharacterEncodingFilter后面，以防乱码(因为它调用了request.getParameter()）
            return registrationBean;
        }

    }

    /**
     * 拦截spring.data.mongodb操作设置在测试环境下使用影子collection
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({TransmittableThreadLocal.class, MongoTemplate.class,Aspect.class})
    public static class SpringDataMongodbPerfConfiguration{

        @Bean
        public SpringDataMongodbPerfAspect getSpringDataMongodbPerfAspect(){
            return new SpringDataMongodbPerfAspect();
        }

    }
    /**
     * 拦截spring.data.mongodb操作设置在测试环境下使用影子collection
     * <br/>
     * 这个类在没有引入perftest包的时候也应该生效，这样没有接入压测的消息消费者才能拒绝执行压测消息
     */
    @Configuration
//    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({TransmittableThreadLocal.class, Consumer.class,Aspect.class})
    public static class OnsPerfConfiguration{
        @Bean
        public OnsPerfAspect getOnsPerfAspect(){
            return new OnsPerfAspect();
        }
    }
    /**
     * 拦截spring.data.mongodb操作设置在测试环境下使用影子collection
     * <br/>
     * 这个类在没有引入perftest包的时候也应该生效，这样没有接入压测的消息消费者才能拒绝执行压测消息
     */
    @Configuration
//    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({TransmittableThreadLocal.class, DefaultMQProducer.class,Aspect.class})
    public static class RocketMqPerfConfiguration{
        @Bean
        public RocketMqPerfAspect getRocketMqPerfAspect(){
            return new RocketMqPerfAspect();
        }
    }

    /**
     * 让CloseableHttpAsyncClient支持线上压测
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass(CloseableHttpAsyncClient.class)
    public static class HttpAsyncClientPostProcessorConfiguration{

        @Bean
        public static SpecifiedBeanPostProcessor httpAsyncClientPostProcessor(){
            return new HttpAsyncClientPostProcessor();
        }
    }

    /**
     * 把spring上下文中的所有数据源转换为自动路由的数据源，可以自动把压测流量导入影子库。
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({AbstractRoutingDataSource.class,TransmittableThreadLocal.class})
    public static class DataSourceConfigurer{

        @Resource
        private ApplicationContext applicationContext;

        @EventListener(EnvironmentChangeEvent.class)
        public void onEnvironmentChange(EnvironmentChangeEvent event) {
            Map<String, PerfTestRoutingDataSource> map = applicationContext.getBeansOfType(PerfTestRoutingDataSource.class);
            if(map == null){
                return;
            }

            map.values().forEach(perfTestRoutingDataSource -> perfTestRoutingDataSource.onEnvironmentChange(event));
        }

        @Bean
        public static SpecifiedBeanPostProcessor perfTestDataSourcePostProcessor(){
            return new SpecifiedBeanPostProcessor<DataSource>() {
                @Autowired
                Environment environment;

                @Autowired
                PerfTestFootMarker perfTestFootMarker;

                @Override
                public int getOrder() {
                    return -2;
                }

                @Override
                public Class<DataSource> getBeanType() {
                    return DataSource.class;
                }

                @Override
                public Object postProcessBeforeInitialization(DataSource bean, String beanName) throws BeansException {
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(DataSource bean, String beanName) throws BeansException {
                    if(isBeanInstanceOfShardingJdbcDataSource(bean, beanName)){
                        return bean;
                    }
                    if(!(bean instanceof PerfTestRoutingDataSource)){
                        PerfTestRoutingDataSource ts = new PerfTestRoutingDataSource(bean, environment, perfTestFootMarker);
                        ts.afterPropertiesSet();
                        bean = ts;
                    }

                    return bean;
                }

            };
        }

        //判断bean是否是sharding-jdbc的datasource,如果是的话，检测内部的数据源是否为PerfTestRoutingDataSource的数据源，如果不是，则报错提示要求声明为spring 的 bean（内部bean会先创建，故肯定会被提前转为PerfTestRoutingDataSource）
        private static boolean isBeanInstanceOfShardingJdbcDataSource(DataSource bean, String beanName){
            try {
                Class.forName("io.shardingjdbc.core.jdbc.adapter.AbstractDataSourceAdapter");
            } catch (ClassNotFoundException e) {
                return false;
            }

            if(!(bean instanceof AbstractDataSourceAdapter)){
                return false;
            }

            //如果是sharding-jdbc的数据源，做些额外校验
            if(bean instanceof ShardingDataSource){
                ShardingDataSource ds = (ShardingDataSource)bean;
                Field field = ReflectionUtils.findField(ShardingDataSource.class, "shardingContext");
                field.setAccessible(true);
                ShardingContext shardingContext = (ShardingContext)ReflectionUtils.getField(field, ds);
                Map<String, DataSource> dataSourceMap = shardingContext.getShardingRule().getDataSourceMap();
                for(Map.Entry<String, DataSource> entry : dataSourceMap.entrySet()){
                    DataSource innerDs = entry.getValue();
                    if(innerDs instanceof MasterSlaveDataSource){
                        processMasterSlaveDataSource((MasterSlaveDataSource)innerDs, beanName);
                    }
                    else if(!(innerDs instanceof PerfTestRoutingDataSource)){
                        onShardingJdbcError(beanName);
                    }
                }
            }else if(bean instanceof MasterSlaveDataSource){
                processMasterSlaveDataSource((MasterSlaveDataSource)bean, beanName);
            }else{
                throw new IllegalStateException("[NOTIFYME]sharding jdbc新增的数据源暂时不支持，如遇到此问题，请联系架构组添加支持");
            }

            return true;
        }

        private static void processMasterSlaveDataSource(DataSource ds, String beanName){//方法签名不使用MasterSlaveDataSource是为了防止报出找不到类的错误
            MasterSlaveDataSource ds1 = (MasterSlaveDataSource)ds;
            MasterSlaveRule oriRule = ds1.getMasterSlaveRule();
            DataSource oriMd = oriRule.getMasterDataSource();

            boolean throwException = false;
            if(!(oriMd instanceof PerfTestRoutingDataSource)){
                throwException = true;
            }
            if(!throwException) {
                Map<String, DataSource> slaveDataSourceMap = oriRule.getSlaveDataSourceMap();
                for (Map.Entry<String, DataSource> entry : slaveDataSourceMap.entrySet()) {
                    DataSource innerDs = entry.getValue();
                    if (!(innerDs instanceof PerfTestRoutingDataSource)) {
                        throwException = true;
                        break;
                    }
                }
            }

            if(throwException){
                onShardingJdbcError(beanName);
            }
        }

        private static void onShardingJdbcError(String beanName){
            throw new IllegalStateException("请把id为["+beanName+"]的sharding-jdbc数据源内部使用的数据源注册为spring的bean，以让线上压测框架有机会处理内部的数据源（内部数据源必须为dbcp2）");
        }

    }

    /**
     * 把spring上下文中的所有Mongoddb数据源转换为自动路由的数据源，可以自动把压测流量导入影子库。
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({MongoProperties.class,TransmittableThreadLocal.class})
    //@ConditionalOnBean({DataSource.class})
    public static class MongoDbDataSourceConfigurer{

        @Bean
        public static SpecifiedBeanPostProcessor perfTestMongoDbDataSourcePostProcessor(){
            return new SpecifiedBeanPostProcessor<MongoProperties>() {
                @Override
                public int getOrder() {
                    return -2;
                }

                @Override
                public Class<MongoProperties> getBeanType() {
                    return MongoProperties.class;
                }

                @Override
                public Object postProcessBeforeInitialization(MongoProperties bean, String beanName) throws BeansException {
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(MongoProperties bean, String beanName) throws BeansException {
                    SpringDataMongodbPerfAspect.OriginDataBaseUri.put(new MongoClientURI(bean.getUri()).getDatabase(),bean.getUri());
                    return bean;
                }
            };
        }

    }

    /**
     * 注册元数据到Eureka中，表示当前cloud应用支持压测。
     * @return
     */
    @Bean
    public DiscoveryMetadataRegister perftestDiscoveryMetadataRegister(){
        return appMetadata -> appMetadata.put(DubboPerfTestRegistryFactoryWrapper.IS_PERF_TEST_SUPPORTTED_KEY,"1");
    }

    /**
     * 配置Feign请求拦截器，在请求前配置http头表示这是压测请求
     */
    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({ HystrixCommand.class, HystrixFeign.class })
    public class PerfTestFeignConfiguration{

        @Bean
        public RequestInterceptor perfTestFeignRequestInterceptor(){
            return template -> {
                if (InternalPerfTestContext.isCurrentInPerfTestMode()) {//表示线上压测流量
                    //在RibbonCustomAutoConfiguration.CustomZoneAvoidanceRule中判断对应服务端是否支持压测，如果不支持压测，会拒绝当前rpc请求
                    template.header(PerfTestUtils.IS_PERF_TEST_MODE_HEADER_KEY, "true");//设置http请求头，传递给服务端，表示是压测请求
                    String sceneId = InternalPerfTestContext.getCurrentSceneId();
                    boolean isTestCluster = InternalPerfTestContext.isTestCluster();
                    if (StringUtils.isNotBlank(sceneId)) {
                        template.header(PerfTestUtils.PERF_TEST_SCENE_ID_KEY, sceneId);  // 设置压测场景id
                        template.header(PerfTestUtils.PERF_TEST_CLUSTER, String.valueOf(isTestCluster));    // 设置是否独立容器集群压测
                    }
                }
            };
        }

    }

    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({ RedisClient.class, Jedis.class, Aspect.class })
    @ConditionalOnBean({ RedisClient.class })
    public static class PerfTestRedisCachePluginConfiguration {

        @Bean
        public PerfTestRedisCachePlugin perfTestRedisCachePlugin() {
            return new PerfTestRedisCachePlugin();
        }

    }

    @Configuration
    @ConditionalOnResource(resources = "/autoconfig/perftest.properties")
    @ConditionalOnClass({ JedisConnectionFactory.class, Jedis.class, Aspect.class})
    public static class PerfTestSpringDataRedisPluginConfiguration{

        @Bean
        public PerfTestSpringDataRedisPlugin perfTestSpringDataRedisPlugin(){
            return new PerfTestSpringDataRedisPlugin();
        }

    }
}
