package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.BufferedHttpResponseWapper;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.HttpClientFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.client.methods.CloseableHttpResponse;

import java.lang.reflect.Method;

/**
 * Created by guoyanfei .
 * 2019-07-18 .
 */
@Slf4j
public class RecordHttpClientMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以录制
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canRecord(Method method, Object[] allArguments) {
        if (!"doExecute".equals(method.getName())) {
            return false;
        }
        if (!FlowReplayTrace.isTraced()) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        if (!canRecord(method, allArguments)) {
            return;
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return;
        }
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(obj, method.getName(), allArguments);
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        if (!canRecord(method, allArguments)) {
            return ret;
        }
        if (!IgnoreSubInvokesContext.isInstMarked(obj, method.getName(), allArguments)) {
            return ret;
        }

        try {
            HttpHost httpHost = (HttpHost) allArguments[0];
            HttpRequest httpRequest = (HttpRequest) allArguments[1];

            if (httpRequest instanceof HttpEntityEnclosingRequest) {
                HttpEntityEnclosingRequest entityEnclosingRequest = (HttpEntityEnclosingRequest) httpRequest;
                HttpEntity httpEntity = entityEnclosingRequest.getEntity();
                // 判断httpEntity中的流是否可以多次读取，如果不能多次读取，那么这个用例不录制，直接返回结果
                if (!httpEntity.isRepeatable()) {
                    FlowReplayTrace.remove();
                    log.debug("{}的entityEnclosingRequest.getEntity不可被多次调用，当前用例取消录制。", HttpClientFlowReplaySpan.parseUrl(httpHost, httpRequest));
                    return ret;
                }
            }

            CloseableHttpResponse response = (CloseableHttpResponse) ret;
            // 包装CloseableHttpResponse，包装后的对象可以多次读取entity流中的内容，否则引流回归读取之后，业务代码中将无法读取
            BufferedHttpResponseWapper responseWapper = new BufferedHttpResponseWapper(response);
            ret = responseWapper;

            HttpClientFlowReplaySpan span = HttpClientFlowReplaySpan.createSpan(httpHost, httpRequest, responseWapper);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
        } catch (Throwable t) {
            log.error("httpClient录制异常", t);
            // 录制异常，这个用例不录了
            FlowReplayTrace.remove();
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // 如果捕捉到异常，并且正在录制中，那么不录这个用例
        if (FlowReplayTrace.isTraced()) {
            FlowReplayTrace.remove();
        }
    }
}
