package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.sampler;

import java.util.BitSet;
import java.util.Random;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 最小支持0.00001F（即十万分之一的采样率）
 * This sampler is appropriate for low-traffic instrumentation (ex servers that each receive <100K
 * requests), or those who do not provision random trace ids. It not appropriate for collectors as
 * the sampling decision isn't idempotent (consistent based on trace id).
 *
 * <h3>Implementation</h3>
 *
 * <p>Taken from <a href="https://github.com/openzipkin/zipkin-java/blob/traceid-sampler/zipkin/src/main/java/zipkin/CountingTraceIdSampler.java">Zipkin project</a></p>
 *
 * <p>This counts to see how many out of 10000 traces should be retained. This means that it is
 * accurate in units of 10000 traces.
 *
 * @author Marcin Grzejszczak
 * @author Adrian Cole
 * @since 1.0.0
 */
public class PercentageBasedRecordSampler implements RecordSampler {

    private final    AtomicInteger counter = new AtomicInteger(0);
    private volatile BitSet        sampleDecisions;
    private volatile float         percentage;

    /**
     * 录制预热60秒内预热完成
     */
    private static final int  WARMUP_TIMESECONDS = 60;

    /**
     * 录制开始时间，每次init的时候需要设置当前时间为录制开始时间，用于预热过程中，采样比例系数增大的判断依据
     */
    private volatile Long recordStartTime;

    /**
     * 采样率的比例系数 比如比例系数是5，则采样率是 0.5 * percentage
     * 最大是10，最小是1
     * 此比例系数从录制开始，根据时间的推移缓慢增大，到达10的时候，表示预热完成
     */
    private volatile int scale;

    public PercentageBasedRecordSampler() {
        init(null, null);
    }

    /**
     * 初始化设置采样率和录制开始时间
     * 采样率默认0.01f
     * 录制开始时间默认当前系统时间
     * @param percentage
     * @param recordStartTime
     */
    public void init(Float percentage, Long recordStartTime) {
        this.percentage = (percentage != null ? percentage : 0.01f);
        this.recordStartTime = (recordStartTime != null ? recordStartTime : System.currentTimeMillis());
        this.scale = 1;
        float currentPercentage = this.percentage * this.scale / 10;
        int outOf100000 = (int) (currentPercentage * 100000.0f);
        this.sampleDecisions = randomBitSet(100000, outOf100000, new Random());
    }

    private void warmup() {
        if (this.scale == 10) { // 比例系数为10，说明已经预热完成
            return;
        }
        int secondsAfterRecordStartTime = (int) ((System.currentTimeMillis() - this.recordStartTime) / 1000); // 系统当前时间距离录制开始时间的秒数
        int newScale = 10 * secondsAfterRecordStartTime / WARMUP_TIMESECONDS + 1;
        newScale = (newScale < 1 ? 1 : (newScale > 10 ? 10 : newScale));    // 计算当前采样率的比例系数
        if (newScale > this.scale) {
            synchronized (this) {
                if (newScale > this.scale) {
                    this.scale = newScale;
                    float currentPercentage = this.percentage * this.scale / 10;
                    int outOf100000 = (int) (currentPercentage * 100000.0f);
                    this.sampleDecisions = randomBitSet(100000, outOf100000, new Random());
                }
            }
        }
    }

    @Override
    public boolean isSampled() {
        if (this.percentage == 0) {  //NOSONAR
            return false;
        } else if (this.percentage == 1.0f && scale == 10) {    //NOSONAR
            return true;
        }
        if (this.scale < 10) { // 比例系数小于10，说明正在预热
            this.warmup();  // 预热
        }
        synchronized (this) {
            final int i = this.counter.getAndIncrement();
            boolean result = this.sampleDecisions.get(i);
            if (i == 99999) {
                this.counter.set(0);
            }
            return result;
        }
    }

    /**
     * Reservoir sampling algorithm borrowed from Stack Overflow.
     * http://stackoverflow.com/questions/12817946/generate-a-random-bitset-with-n-1s
     */
    static BitSet randomBitSet(int size, int cardinality, Random rnd) {
        BitSet result = new BitSet(size);
        int[] chosen = new int[cardinality];
        int i;
        for (i = 0; i < cardinality; ++i) {
            chosen[i] = i;
            result.set(i);
        }
        for (; i < size; ++i) {
            int j = rnd.nextInt(i + 1);
            if (j < cardinality) {
                result.clear(chosen[j]);
                result.set(i);
                chosen[j] = i;
            }
        }
        return result;
    }

}
