package cn.com.duibaboot.ext.autoconfigure.batch;

import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.TimeoutException;

/**
 * 拦截器，用来拦截方法上注解了 @MergeRequest 的方法
 *
 * @author houwen
 */
@Aspect
public class ReqInterceptor {

    @Resource
    private BatchProperties properties;

    @Resource
    private ReqBucket reqBucket;

    @Around("@annotation(cn.com.duibaboot.ext.autoconfigure.batch.MergeRequest)")
    public Object merge(ProceedingJoinPoint joinPoint) throws Throwable {
        if (!properties.isEnable()) {
            return joinPoint.proceed();
        }
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        MergeRequest merge = AnnotationUtils.getAnnotation(signature.getMethod(), MergeRequest.class);
        if (merge == null) {
            return joinPoint.proceed();
        }
        if (StringUtils.isEmpty(merge.method())) {
            return joinPoint.proceed();
        }
        ReqContext req = new ReqContext(joinPoint.getArgs(), joinPoint.getTarget(), merge);
        Object ret;
        try {
            ret = submitRequest(req);
        } catch (RejectedExecutionException e) {
            // RejectedExecutionException：代表当前线程池繁忙 走原有流程
            return joinPoint.proceed();
        }
        return ret;
    }

    private Object submitRequest(ReqContext req) throws Throwable {
        reqBucket.submit(req);
        //异步任务直接返回
        if (req.getAnnotation().oneway()) {
            return null;
        }
        //等待处理的返回结果
        synchronized (req) {
            req.wait(req.getAnnotation().timeout());
        }
        if (req.getState() != ReqContext.NOTIFY) {
            //没有通知，需要把超时异常抛到上层
            throw new TimeoutException(req.getFullMethodName() + " " + req.getAnnotation().timeout() + "ms");
        }
        return req.getResult();
    }

}
