package cn.com.duibaboot.ext.autoconfigure.cloud.zipkin;

import ch.qos.logback.classic.LoggerContext;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.cloud.sleuth.Sampler;
import org.springframework.cloud.sleuth.Tracer;
import org.springframework.cloud.sleuth.instrument.web.TraceFilter;
import org.springframework.cloud.sleuth.sampler.SamplerProperties;
import org.springframework.cloud.sleuth.zipkin2.ZipkinAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurerAdapter;

import javax.servlet.Filter;

/**
 * 设置采样器（最低支持万分之一的采样率）
 */
@Configuration
@ConditionalOnClass({zipkin2.Span.class, ZipkinAutoConfiguration.class})
@EnableConfigurationProperties({SamplerProperties.class})
@AutoConfigureBefore(ZipkinAutoConfiguration.class)
@ConditionalOnProperty(value = "spring.zipkin.enabled", matchIfMissing = true)
public class DuibaZipkinAutoConfiguration extends WebMvcConfigurerAdapter {

    @Autowired
    private DuibaRpcParamsInterceptor interceptor;

    @Configuration
    @ConditionalOnClass(RefreshScope.class)
    protected static class RefreshScopedPercentageBasedSamplerConfiguration {
        @Bean
        @RefreshScope
        @ConditionalOnMissingBean
        public Sampler defaultTraceSampler(SamplerProperties config) {
            return new PercentageBasedSampler(config);
        }
    }

    @Configuration
    @ConditionalOnMissingClass("org.springframework.cloud.context.config.annotation.RefreshScope")
    protected static class NonRefreshScopePercentageBasedSamplerConfiguration {
        @Bean
        @ConditionalOnMissingBean
        public Sampler defaultTraceSampler(SamplerProperties config) {
            return new PercentageBasedSampler(config);
        }
    }

    @Bean
    @ConditionalOnClass(TraceFilter.class)
    public Filter duibaTraceFilter() {

        LoggerContext context = (LoggerContext)LoggerFactory.getILoggerFactory();
        context.addTurboFilter(new ZipkinLogTurboFilter());

        return new DuibaTraceFilter();
    }

    @Bean
    @ConditionalOnClass({TraceFilter.class,Tracer.class})
    public Filter duibaTracSecondFilter() {
        return new DuibaTraceSecondFilter();
    }

    @Bean
    @ConditionalOnClass({TraceFilter.class,Tracer.class})
    public DuibaRpcParamsInterceptor duibaRpcParamsInterceptor() {
        return new DuibaRpcParamsInterceptor();
    }

    @Override
    @ConditionalOnClass({TraceFilter.class,Tracer.class})
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(interceptor).addPathPatterns("/**");
    }

    @Bean
    @ConditionalOnProperty(value = "duiba.zipkin.port", matchIfMissing = false)
    public static SpecifiedBeanPostProcessor zipkinLoadBalancerBeanPostProcessor(@Value("${duiba.zipkin.port}") int zipkinPort) {
        return new ZipkinLoadBalancerBeanPostProcessor(zipkinPort);
    }
}
