package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayErrorMsgTypeEnum;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.exception.ExceptionUtils;

import java.io.Serializable;
import java.util.HashSet;
import java.util.Set;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicInteger;

/**
 * 回归的单个trace上下文
 * Created by guoyanfei .
 * 2019-02-27 .
 */
@Slf4j
public class ReplayTraceContext implements Serializable {

    private static final long serialVersionUID = 2716578641071393051L;

    private String traceId;

    private FlowReplaySpan mainSpan;

    private LinkedBlockingQueue<FlowReplaySpan> subSpans;

    /**
     * 标记了用例失败
     */
    private boolean errorMarked;

    /**
     * 错误原因
     */
    private FlowReplayErrorMsgTypeEnum errorMsgType;

    /**
     * 错误详情
     */
    private String errorMsgDetail;

    /**
     * 中间步骤对比不一致字段list
     */
    private Set<StepDiffColumn> stepDiffColumns;

    /**
     * 回归过程当前span序号，从0开始
     */
    private AtomicInteger currentSpanIdx;

    /**
     * 回归详情的trace，用于生成回归详情文件
     */
    private FlowReplayTrace replayDetailTrace;

    private static class ThreadLocalHolder {

        private ThreadLocalHolder() {
        }

        private static final ThreadLocal<ReplayTraceContext> threadLocal2ReplayTraceContext = new ThreadLocal<>();
    }

    /**
     * 创建trace
     * @param trace
     * @param replayDetailTrace
     */
    public static void create(FlowReplayTrace trace, FlowReplayTrace replayDetailTrace) {
        ReplayTraceContext c = new ReplayTraceContext();
        c.traceId = trace.getTraceId();
        c.mainSpan = trace.getMainSpan();
        c.subSpans = new LinkedBlockingQueue<>(trace.getSubSpans());
        c.errorMarked = false;
        c.errorMsgType = null;
        c.errorMsgDetail = null;
        c.replayDetailTrace = replayDetailTrace;
        c.stepDiffColumns = new HashSet<>();
        c.currentSpanIdx = new AtomicInteger(0);
        ThreadLocalHolder.threadLocal2ReplayTraceContext.set(c);
    }

    /**
     * 把回归中间步骤对比不一致字段信息，加入到队列中
     * @param stepDiffColumn
     */
    public static void addAllStepDiffColumns(Set<StepDiffColumn> stepDiffColumns) {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null || c.stepDiffColumns == null) {
            return;
        }
        for (StepDiffColumn it : stepDiffColumns) {
            // 判断对比不一致的字段是否是需要屏蔽的
            if (!ReplayContextHolder.isMaskedStepDiffColumn(it)) {
                c.stepDiffColumns.add(it);
            }
        }
    }

    /**
     * 标记本次回归请求为异常
     * @param errorMsgType
     * @param expert
     * @param actual
     * @param exceptionMsg
     */
    private static void markError(FlowReplayErrorMsgTypeEnum errorMsgType, String expert, String actual, Throwable t) {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null || c.errorMarked) {   // context未创建 || 已经被标记过
            return;
        }
        c.errorMarked = true;
        c.errorMsgType = errorMsgType;
        StringBuilder sb = new StringBuilder();
        if (StringUtils.isNotBlank(expert) || StringUtils.isNotBlank(actual)) {
            sb.append(String.format("期望: %s, 实际: %s \n", expert, actual));
        }
        sb.append(ExceptionUtils.getStackTrace(t != null ? t : new Exception()));
        c.errorMsgDetail = sb.toString();
    }

    /**
     * 获取兑吧的堆栈信息
     * @param t
     * @return
     */
    private static String getDuibaStackTrace(Throwable t) {
        String[] stackFrames = ExceptionUtils.getStackFrames(t);
        StringBuilder sb = new StringBuilder();
        for (String it : stackFrames) {
            if (it.startsWith("\tat ") && !it.startsWith("\tat cn.com.duiba")) {
                continue;
            }
            sb.append(it);
            sb.append("\n");
        }
        return sb.toString();
    }

    /**
     * 标记本次回归请求为异常
     * @param errorMsgType
     */
    public static void markError(FlowReplayErrorMsgTypeEnum errorMsgType) {
        markError(errorMsgType, null, null, null);
    }

    /**
     * 标记本次回归请求为异常
     * @param errorMsgType
     * @param expert
     * @param actual
     */
    public static void markError(FlowReplayErrorMsgTypeEnum errorMsgType, String expert, String actual) {
        markError(errorMsgType, expert, actual, null);
    }

    /**
     * 标记本次回归请求为异常
     * @param errorMsgType
     * @param t
     */
    public static void markError(FlowReplayErrorMsgTypeEnum errorMsgType, Throwable t) {
        markError(errorMsgType, null, null, t);
    }

    /**
     * 获取并删除span队列头部的span
     * @return
     */
    public static FlowReplaySpan pollSubSpan() {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null || c.subSpans == null || c.currentSpanIdx == null) {
            return null;
        }
        c.currentSpanIdx.incrementAndGet();
        return c.subSpans.poll();
    }

    /**
     * 获取当前回归到的span的序号
     * @return
     */
    public static Integer getCurrentSpanIdx() {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null || c.currentSpanIdx == null) {
            return null;
        }
        return c.currentSpanIdx.get();
    }

    /**
     * 获取并删除当前回归请求上下文
     * @return
     */
    public static ReplayTraceContext getAndRemove() {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null) {
            return null;
        }
        ThreadLocalHolder.threadLocal2ReplayTraceContext.remove();
        return c;
    }

    public static ReplayTraceContext get() {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null) {
            return null;
        }
        return c;
    }

    public static String getContextTraceId() {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null) {
            return null;
        }
        return c.traceId;
    }

    public static void remove() {
        ThreadLocalHolder.threadLocal2ReplayTraceContext.remove();
    }

    /**
     * 是否正在回归(本次请求是否是回归请求)
     * @return
     */
    public static boolean isReplaying() {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        return c != null;
    }

    /**
     * 获取完整的失败信息
     * @return
     */
    public static String getCompletedErrorMsg() {
        ReplayTraceContext c = ThreadLocalHolder.threadLocal2ReplayTraceContext.get();
        if (c == null) {
            return null;
        }
        if (c.errorMsgType == null) {
            return null;
        }
        return c.errorMsgType.desc()+ ",traceId=" + c.traceId + "," + c.errorMsgDetail;
    }

    public String getTraceId() {
        return traceId;
    }

    public FlowReplaySpan getMainSpan() {
        return mainSpan;
    }

    public LinkedBlockingQueue<FlowReplaySpan> getSubSpans() {
        return subSpans;
    }

    public boolean isErrorMarked() {
        return errorMarked;
    }

    public FlowReplayErrorMsgTypeEnum getErrorMsgType() {
        return errorMsgType;
    }

    public void setTraceId(String traceId) {
        this.traceId = traceId;
    }

    public String getErrorMsgDetail() {
        return errorMsgDetail;
    }

    public FlowReplayTrace getReplayDetailTrace() {
        return replayDetailTrace;
    }

    public Set<StepDiffColumn> getStepDiffColumns() {
        return stepDiffColumns;
    }
}
