package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayCloseableHttpResponse;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.HttpClientFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.SpanType;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;
import org.apache.http.ProtocolVersion;
import org.apache.http.StatusLine;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.message.BasicStatusLine;

import java.io.IOException;
import java.lang.reflect.Method;
import java.net.URI;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * Created by guoyanfei .
 * 2019-07-18 .
 */
@Slf4j
public class ReplayHttpClientMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以回归
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        if (!"doExecute".equals(method.getName())) {
            return false;
        }
        // 当前不是回归的环境 或者 当前请求不是回归请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        if (!canReplay(method, allArguments)) {
            return;
        }

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        HttpHost httpHost = (HttpHost) allArguments[0];
        HttpRequest httpRequest = (HttpRequest) allArguments[1];

        // 创建回归的span，用于回归详情的构建
        HttpClientFlowReplaySpan replayDetailSpan = createReplayDetailSpan(httpHost, httpRequest, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.HTTP_CLIENT != span.getSpanType()) {
            String expert = SpanType.HTTP_CLIENT.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        HttpClientFlowReplaySpan httpClientSpan = (HttpClientFlowReplaySpan) span;

        String currentHttpMethod = HttpClientFlowReplaySpan.parseMethod(httpRequest);
        String currentUrl = HttpClientFlowReplaySpan.parseUrl(httpHost, httpRequest);
        Map<String, List<String>> currentParameters = HttpClientFlowReplaySpan.parseRequestParameters(httpRequest);
        byte[] currentRequestBody = HttpClientFlowReplaySpan.parseRequestBody(httpRequest);

        if (!httpClientSpan.getMethod().equals(currentHttpMethod)) {
            String expert = httpClientSpan.getMethod();
            String actual = currentHttpMethod;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1011, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        URI uriRecord = new URI(httpClientSpan.getUrl());
        URI uriReplay = new URI(currentUrl);
        if (!uriRecord.getPath().equals(uriReplay.getPath()) || !Objects.equals(uriRecord.getQuery(), uriReplay.getQuery())) {
            String expert = httpClientSpan.getUrl();
            String actual = currentUrl;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1012, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!isParametersEqual(httpClientSpan.getRequestParameters(), currentParameters)) {
            String expert = httpClientSpan.getRequestParameters() != null ? httpClientSpan.getRequestParameters().toString() : null;
            String actual = currentParameters.toString();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1013, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!Arrays.equals(httpClientSpan.getRequestBody(), currentRequestBody)) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1014);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        ProtocolVersion version = new ProtocolVersion(httpClientSpan.getProtocol(), httpClientSpan.getMajor(), httpClientSpan.getMinor());
        StatusLine statusLine = new BasicStatusLine(version, httpClientSpan.getStatusCode(), httpClientSpan.getReasonPhrase());
        CloseableHttpResponse mockResp = new ReplayCloseableHttpResponse(httpClientSpan.getResponseBody(), httpClientSpan.getResponseHeaders(), statusLine);

        BufferedHttpResponseWapper mockRespWapper = new BufferedHttpResponseWapper(mockResp);
        // 参数都相同的情况下，mock之前把mock的返回值回填到构建回归详情用的span中
        replayDetailSpan.setResp(mockRespWapper);

        result.defineReturnValue(mockRespWapper);
    }

    private HttpClientFlowReplaySpan createReplayDetailSpan(HttpHost httpHost, HttpRequest httpRequest, FlowReplaySpan span) {
        HttpClientFlowReplaySpan replayDetailSpan = null;
        try {
            replayDetailSpan = HttpClientFlowReplaySpan.createSpan(httpHost, httpRequest, null);
            replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
            if (span != null) {
                replayDetailSpan.setSpanId(span.getSpanId());
            }
            FlowReplayTrace.addSubSpan(replayDetailSpan);
        } catch (IOException e) {
            throw new PluginException("正常这个异常不可能出来...HttpClientFlowReplaySpan.createSpan(),第三个参数不为空才有可能", e);
        }

        return replayDetailSpan;
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // do nothing
    }

    private boolean isParametersEqual(Map<String, List<String>> recordParameters, Map<String, List<String>> currentParameters) {
        Preconditions.checkNotNull(recordParameters, "recordParameters can't be null");
        Preconditions.checkNotNull(currentParameters, "currentParameters can't be null");
        if (recordParameters.size() != currentParameters.size()) {
            return false;
        }
        for (Map.Entry<String, List<String>> entry : recordParameters.entrySet()) {
            String k = entry.getKey();
            List<String> v = entry.getValue();
            List<String> cv = currentParameters.get(k);
            if (v == null && cv == null) {
                continue;
            }
            if (v == null || cv == null) {
                return false;
            }
            if (!CollectionUtils.isEqualCollection(v, cv)) {
                return false;
            }
        }
        return true;
    }
}
