package cn.com.duibaboot.ext.autoconfigure.ons;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duibaboot.ext.autoconfigure.core.EarlyClose;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import com.aliyun.openservices.ons.api.*;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * aliyun ONS消息队列生产者/消费者自动配置
 * Created by wenqi.huang on 2016/12/26.
 */
@Configuration
@ConditionalOnClass({Producer.class})
@EnableConfigurationProperties(OnsProperties.class)
public class OnsAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(OnsAutoConfiguration.class);

    protected abstract static class OnsClientConfiguration{

        @Autowired
        protected OnsProperties onsProperties;

    }

    @Configuration
    @ConditionalOnClass({Producer.class})
    @ConditionalOnMissingBean(name={"onsProducer"})
    @ConditionalOnProperty(name="duiba.ons.producer.enable", havingValue = "true", matchIfMissing = false)
    protected static class OnsProducerConfigurator extends OnsClientConfiguration{

        @Bean(name="onsProducer", destroyMethod = "shutdown")
        public Producer onsProducer(){
            Properties properties = new Properties();
            String producerId = onsProperties.getProducer().getGroup();
            if(producerId != null && !producerId.equals("")) {
                properties.put(PropertyKeyConst.ProducerId, producerId);// 您在MQ控制台创建的Producer ID
            }
            properties.put(PropertyKeyConst.AccessKey, onsProperties.getAccessKey());// 鉴权用AccessKey
            properties.put(PropertyKeyConst.SecretKey, onsProperties.getSecretKey());// 鉴权用SecretKey
            properties.put(PropertyKeyConst.SendMsgTimeoutMillis, onsProperties.getProducer().getSendMsgTimeoutMillis());// 设置发送超时时间，单位毫秒
            //公有云生产环境：http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //公有云公测环境：http://onsaddr-internet.aliyun.com/rocketmq/nsaddr4client-internet
            //杭州金融云环境：http://jbponsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //杭州深圳云环境：http://mq4finance-sz.addr.aliyun.com:8080/rocketmq/nsaddr4client-internal
//            properties.put(PropertyKeyConst.ONSAddr,
//                    "http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal");//此处以公有云生产环境为例
            if(!StringUtils.isBlank(onsProperties.getNameSrvAddr())
                    && !StringUtils.equals("ons", onsProperties.getNameSrvAddr())) {
                properties.put(PropertyKeyConst.NAMESRV_ADDR, onsProperties.getNameSrvAddr());
            }
            Producer producer = ONSFactory.createProducer(properties);
            producer = new ProducerWrapper(producer);//包装一下

            // 在发送消息前，必须调用start方法来启动Producer，只需调用一次即可
            producer.start();

            return producer;
        }
    }

    @Configuration
    @ConditionalOnClass({Consumer.class})
    @ConditionalOnMissingBean(name={"onsConsumer"})
    @ConditionalOnProperty(name="duiba.ons.consumer.enable", havingValue = "true", matchIfMissing = false)
    protected static class OnsConsumerConfigurator extends OnsClientConfiguration{

        /**
         * 对MessageListener进行包装，增加cat监控
         * @return
         */
        @Bean
        public static SpecifiedBeanPostProcessor<MessageListener> onsMessageListenerPostProcessor(){
            return new SpecifiedBeanPostProcessor<MessageListener>() {
                @Override
                public Class<MessageListener> getBeanType() {
                    return MessageListener.class;
                }

                @Override
                public Object postProcessBeforeInitialization(MessageListener bean, String beanName) throws BeansException {
                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(MessageListener bean, String beanName) throws BeansException {
                    return new OnsMessageListenerWrapper(bean);
                }

                @Override
                public int getOrder() {
                    return 0;
                }
            };
        }

        @Bean(name="onsConsumer")
        public FactoryBean<Consumer> onsConsumer() {
            return new OnsConsumerFactoryBean(onsProperties);
        }

        @EventListener(MainContextRefreshedEvent.class)
        @Order(Ordered.LOWEST_PRECEDENCE)//用户应用可能要加载缓存
        public void onEvent(MainContextRefreshedEvent event){
            if(FlowReplayUtils.isReplayEnv()){//流量回归实例不开启消费
                return;
            }
            Map<String, Consumer> map = event.getApplicationContext().getBeansOfType(Consumer.class);
            if(!map.isEmpty()){
                for(Consumer consumer : map.values()){
                    //Consumer对象在使用之前必须要调用start初始化，初始化一次即可
                    if(consumer != null) {
                        consumer.start();
                    }
                }
            }
        }
    }

    private static class OnsConsumerFactoryBean extends EarlyClose implements FactoryBean<Consumer>,InitializingBean{
        @Autowired(required = false)
        @Qualifier("bootOnsMessageListener")
        private MessageListener onsMessageListener;
        private Consumer onsConsumer;
        private OnsProperties onsProperties;

        public OnsConsumerFactoryBean(OnsProperties onsProperties){
            this.onsProperties = onsProperties;
        }

        @Override
        public void afterPropertiesSet() throws Exception {
            if(onsMessageListener == null){
                logger.error("", new IllegalStateException("bean id:[bootOnsMessageListener] class:[com.aliyun.openservices.ons.api.MessageListener] is not exist in spring's context, 请声明，否则不会启用ons消费!!!,请参考： http://cf.dui88.com/pages/viewpage.action?pageId=4493915"));
                return;
            }
            Properties properties = new Properties();
            properties.put(PropertyKeyConst.ConsumerId, onsProperties.getConsumer().getGroup());// 您在MQ控制台创建的Consumer ID
            properties.put(PropertyKeyConst.AccessKey, onsProperties.getAccessKey());// 鉴权用AccessKey，在阿里云服务器管理控制台创建
            properties.put(PropertyKeyConst.SecretKey, onsProperties.getSecretKey());// 鉴权用SecretKey，在阿里云服务器管理控制台创建
            properties.put(PropertyKeyConst.MaxReconsumeTimes, onsProperties.getConsumer().getMaxReconsumeTimes());// 消息消费失败时的最大重试次数,默认值16
            if(!StringUtils.isBlank(onsProperties.getNameSrvAddr())
                && !StringUtils.equals("ons", onsProperties.getNameSrvAddr())) {
                properties.put(PropertyKeyConst.NAMESRV_ADDR, onsProperties.getNameSrvAddr());
            }

            if(onsProperties.getConsumer().getConsumeThreadNums() > 0) {
                properties.put(PropertyKeyConst.ConsumeThreadNums, onsProperties.getConsumer().getConsumeThreadNums());// 消费线程数量
            }

            //公有云生产环境：http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //公有云公测环境：http://onsaddr-internet.aliyun.com/rocketmq/nsaddr4client-internet
            //杭州金融云环境：http://jbponsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //杭州深圳云环境：http://mq4finance-sz.addr.aliyun.com:8080/rocketmq/nsaddr4client-internal
            //properties.put(PropertyKeyConst.ONSAddr, "http://onsaddr-internal.aliyun.com:8080/rocketmq/nsaddr4client-internal");//此处以公有云生产环境为例
            com.aliyun.openservices.ons.api.Consumer consumer = ONSFactory.createConsumer(properties);
            String topics = onsProperties.getConsumer().getTopics();
            String[] topicArr = topics.split(",");

            for(String topic : topicArr) {
                consumer.subscribe(topic, "*", onsMessageListener);
            }

            onsConsumer = consumer;
        }

        @Override
        public Consumer getObject() throws Exception {
            return onsConsumer;
        }

        @Override
        public Class<?> getObjectType() {
            return Consumer.class;
        }

        @Override
        public boolean isSingleton() {
            return true;
        }

        @Override
        public void stop() {
            if(onsConsumer != null) {
                onsConsumer.shutdown();
                awaitShutdown(onsConsumer, 3);
            }
        }

        /**
         * 通过反射取出主线程池并等待一会,优雅停机
         * @param onsConsumer
         * @param awaitMaxSeconds
         */
        private void awaitShutdown(Consumer onsConsumer, int awaitMaxSeconds){
            Field fieldDefaultMQPushConsumer = ReflectionUtils.findField(onsConsumer.getClass(), "defaultMQPushConsumer");
            fieldDefaultMQPushConsumer.setAccessible(true);
            Object defaultMQPushConsumer = ReflectionUtils.getField(fieldDefaultMQPushConsumer, onsConsumer);

            Field fieldDefaultMQPushConsumerImpl = ReflectionUtils.findField(defaultMQPushConsumer.getClass(), "defaultMQPushConsumerImpl");
            fieldDefaultMQPushConsumerImpl.setAccessible(true);
            Object defaultMQPushConsumerImpl = ReflectionUtils.getField(fieldDefaultMQPushConsumerImpl, defaultMQPushConsumer);

            Field fieldConsumeMessageService = ReflectionUtils.findField(defaultMQPushConsumerImpl.getClass(), "consumeMessageService");
            fieldConsumeMessageService.setAccessible(true);
            Object consumeMessageService = ReflectionUtils.getField(fieldConsumeMessageService, defaultMQPushConsumerImpl);

            Field fieldConsumeExecutor = ReflectionUtils.findField(consumeMessageService.getClass(), "consumeExecutor");
            fieldConsumeExecutor.setAccessible(true);
            ThreadPoolExecutor consumeExecutor = (ThreadPoolExecutor)ReflectionUtils.getField(fieldConsumeExecutor, consumeMessageService);

            try {
                consumeExecutor.awaitTermination(awaitMaxSeconds, TimeUnit.SECONDS);
            } catch (InterruptedException e) {
                //Ignore
                Thread.currentThread().interrupt();
            }
        }

        //和rocketMqConsumer一起关闭，且在统一线程池之前关闭
        @Override
        public int getPhase() {
            return 1;
        }
    }
}
