package cn.com.duibaboot.ext.autoconfigure.perftest;

import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.dbcp2.BasicDataSource;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.env.Environment;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.util.Assert;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * 自动根据当前是不是压测请求决定访问原始数据库或者影子库
 * Created by wenqi.huang on 2016/11/24.
 */
@Slf4j
public class PerfTestRoutingDataSource extends AbstractRoutingDataSource implements AutoCloseable {

    /**
     * 原始库的key
     */
    private static final String ORIGINAL_DATASOURCE = "original";
    /**
     * 影子库的key
     */
    private static final String TEST_DATASOURCE = "test";
    /**
     * 影子库的scheme前缀
     */
    public static final String TEST_SCHEME_PREFIX = "perf__";
    private static final String SHADE_SCHEMA = ".shade";

    private volatile boolean hasTestDataSource = false;
    private volatile boolean testDataSourceInitted = false;
    /**
     * 原始库的scheme
     */
    private String originalScheme;

    /**
     * 原始库datasource
     */
    private BasicDataSource originalDataSource;
    private Environment environment;
    private volatile BasicDataSource shadeDataSource;
    private ConcurrentMap<Object, Object> targetDataSources = new ConcurrentHashMap<>();
    private boolean thisInitted = false;
    private ConcurrentMap<String, String> normalDbUrl2shadeUrlMap = new ConcurrentHashMap<>();
    private List<String> dbShadeKeys = Collections.synchronizedList(new ArrayList<>());
    private PerfTestFootMarker perfTestFootMarker;

    public PerfTestRoutingDataSource(DataSource originalDataSource, Environment environment, PerfTestFootMarker perfTestFootMarker) {
        if(originalDataSource instanceof BasicDataSource){//dbcp2
            this.originalDataSource = (BasicDataSource)originalDataSource;
            this.environment = environment;
            this.perfTestFootMarker = perfTestFootMarker;
        }else{
            throw new IllegalStateException("检测到有DataSource没有使用dbcp2，请先改成dbcp2。");
            //logger.error("检测到有DataSource没有使用dbcp2，请先改成dbcp2。");
        }
    }

    @Override
    public void afterPropertiesSet() {
        if(thisInitted){
            return;
        }
        thisInitted = true;

        targetDataSources.put(ORIGINAL_DATASOURCE, originalDataSource);
        String originalUrl = (originalDataSource).getUrl();

        originalScheme = getOriginalScheme(originalUrl);

        super.setTargetDataSources(targetDataSources);
        initShadeDbMap();
        super.afterPropertiesSet();
    }

    private void initShadeDbMap() {
        LinkedHashMap<String, Object> properties = SpringEnvironmentUtils.getFlatEnvironments(environment);
        properties.entrySet().stream().forEach(x -> {
            if (x.getKey().endsWith(SHADE_SCHEMA)) {
                String dbUrl = environment.getProperty(x.getKey().replace(SHADE_SCHEMA, ""));
                if (originalDataSource.getUrl().equals(dbUrl)) {//是当前数据源的影子配置时才加入
                    normalDbUrl2shadeUrlMap.put(dbUrl, x.getValue().toString());
                    if (!dbShadeKeys.contains(x.getKey())) {
                        dbShadeKeys.add(x.getKey());
                    }
                }
            }
        });
    }

    /**
     * 根据原始数据源得到对应的影子DataSource，影子数据库不存在则返回null.
     * 按照约定，影子库url和原始库url一致，只是scheme部分加入前缀perf__,用户名密码也必须相同，同一个用户要能看到这两个库
     * @param originalDataSource
     */
    private BasicDataSource getTestDataSource(BasicDataSource originalDataSource){
        String testUrl = normalDbUrl2shadeUrlMap.get(originalDataSource.getUrl());//如果存在影子库单独配置则优先使用该配置

        if(StringUtils.isBlank(testUrl) && !StringUtils.isBlank(originalScheme)) {
            Set<String> schemes = new HashSet<>();
            //用show databases命令检测所有数据库scheme，目前只支持mysql数据库的检测
            try (
                Connection connection = originalDataSource.getConnection();
                Statement statement = connection.createStatement();
                ResultSet rs = statement.executeQuery("show databases");
            ){
                while (rs.next()) {
                    schemes.add(rs.getString(1));
                }
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
            log.info("schemes of originalScheme:{}", schemes);
            String testScheme = TEST_SCHEME_PREFIX + originalScheme;
            if(schemes.contains(testScheme)){//影子库存在
                testUrl = getTestUrl(originalDataSource.getUrl());
            }
        }

        if(testUrl == null){//影子库不存在
            return null;
        }

        BasicDataSource ds = new BasicDataSource();
        ds.setDriverClassName(originalDataSource.getDriverClassName());
        ds.setUrl(testUrl);
        ds.setUsername(originalDataSource.getUsername());
        ds.setPassword(originalDataSource.getPassword());
        ds.setDefaultQueryTimeout(originalDataSource.getDefaultQueryTimeout());
        ds.setInitialSize(originalDataSource.getInitialSize());
        ds.setLogAbandoned(originalDataSource.getLogAbandoned());
        ds.setMaxIdle(originalDataSource.getMaxIdle());
        ds.setMaxTotal(originalDataSource.getMaxTotal());
        ds.setMinIdle(0);//压测连接池最小空闲设为0，这样可以在没有压测行为的时候尽快释放压测连接；
        ds.setValidationQuery(originalDataSource.getValidationQuery());
        ds.setValidationQueryTimeout(originalDataSource.getValidationQueryTimeout());
        ds.setTimeBetweenEvictionRunsMillis(30000);//压测连接池每隔30秒扫描一次，清理空闲链接
        ds.setSoftMinEvictableIdleTimeMillis(30000);//清理所有空闲了30秒的压测连接，只保留minIdle个连接
        ds.setTestWhileIdle(originalDataSource.getTestWhileIdle());
        ds.setTestOnBorrow(originalDataSource.getTestOnBorrow());
        ds.setTestOnReturn(originalDataSource.getTestOnReturn());
        ds.setTestOnCreate(originalDataSource.getTestOnCreate());
        ds.setMaxWaitMillis(originalDataSource.getMaxWaitMillis());
        ds.setNumTestsPerEvictionRun(originalDataSource.getNumTestsPerEvictionRun());
        ds.setRemoveAbandonedOnBorrow(originalDataSource.getRemoveAbandonedOnBorrow());
        ds.setRemoveAbandonedOnMaintenance(originalDataSource.getRemoveAbandonedOnMaintenance());
        ds.setRemoveAbandonedTimeout(originalDataSource.getRemoveAbandonedTimeout());

        return ds;
    }

    /**
     * 根据原始的数据源url得到对应的影子库url，按照约定，影子库url和原始库url一致，只是scheme部分加入前缀perf__
     * @param originalUrl
     * @return
     */
    private String getTestUrl(String originalUrl){
        int index = originalUrl.lastIndexOf('/');
        if(index > 0){
            return originalUrl.substring(0, index + 1) + TEST_SCHEME_PREFIX + originalUrl.substring(index+1, originalUrl.length());
        }else{
            throw new IllegalArgumentException(originalUrl + " is invalid");
        }
    }

    /**
     * 根据原始的数据源url得到它的scheme
     * @param originalUrl
     * @return
     */
    private String getOriginalScheme(String originalUrl){
        if(originalUrl.startsWith("jdbc:phoenix")){//hbase phoenix没有scheme，只支持设置.shade
            return null;
        }

        int index = originalUrl.indexOf("//");
        if(index > 0) {
            originalUrl = org.apache.commons.lang3.StringUtils.substring(originalUrl, index + 2);
        }

        index = originalUrl.indexOf('/');
        if(index > 0){
            String schemeAfter = originalUrl.substring(index+1, originalUrl.length());

            int index1 = schemeAfter.indexOf('?');
            if(index1 > 0){
                return schemeAfter.substring(0, index1);
            }else{
                return schemeAfter;
            }
        }else{
            throw new IllegalArgumentException(originalUrl + " is invalid");
        }
    }

    @Override
    protected DataSource determineTargetDataSource() {
        Assert.notNull(this.targetDataSources, "DataSource router not initialized");
        Object lookupKey = determineCurrentLookupKey();
        DataSource dataSource = (DataSource)this.targetDataSources.get(lookupKey);

        if (dataSource == null) {
            throw new IllegalStateException("Cannot determine target DataSource for lookup key [" + lookupKey + "]");
        }
        return dataSource;
    }

    /**
     * 调用refresh后配置变更,重新初始化测试库
     */
//    @EventListener(EnvironmentChangeEvent.class)//这个注解不会生效，当前方法会被ext的其他地方调用
    public void onEnvironmentChange(EnvironmentChangeEvent event) {
        if(event.getKeys().stream().filter(s -> s.endsWith(SHADE_SCHEMA)).count() > 0){
            initShadeDbMap();
            if (testDataSourceInitted) {
                testDataSourceInitted = false;
            }
        } else if(testDataSourceInitted && !hasTestDataSource){
            //刷新后如果发现刷新前没有找到影子库，则重置标记, 让后续重新寻找影子库是否存在.
            testDataSourceInitted = false;
        }
    }

    protected Object determineCurrentLookupKey(){
        if(PerfTestContext.isCurrentInPerfTestMode()){//压测时，就算压测影子库不存在也不走原始库，防止产生脏数据
            if(!testDataSourceInitted){//在这里延迟加载
                initTestDataSource();
            }

            if(!hasTestDataSource){
                //压测时找不到影子库的异常抛出HystrixBadRequestException，以防触发客户端熔断
                throw new HystrixBadRequestException("", new IllegalStateException(originalDataSource.getUrl() + "的影子库不存在，放弃本次请求（如果在创建影子库后仍然遇到此异常，请在棱镜进行【刷新配置】操作后再试）"));
            }
            PerfTestContext.debugInfo("PerfTest DB");

            perfTestFootMarker.markDb(shadeDataSource.getUrl());
            return TEST_DATASOURCE;
        }

        return ORIGINAL_DATASOURCE;
    }

    /**
     * 初始化影子库的DataSource，如果找不到影子库，则为null
     */
    private void initTestDataSource(){
        synchronized (this){
            if(!testDataSourceInitted) {
                BasicDataSource testDataSource = getTestDataSource(originalDataSource);

                if (testDataSource != null) {
                    hasTestDataSource = true;
                    shadeDataSource = testDataSource;
                    targetDataSources.put(TEST_DATASOURCE, testDataSource);//替换
                    log.info("检测到对数据库：{}的压测行为，已初始化压测专用数据库连接池", originalScheme);
                } else {
                    logger.warn(originalDataSource.getUrl() + " 没有对应的影子数据库（在同个库中以perf__开头的scheme, 或者添加.shade配置），如果你配置了，需要【刷新配置】生效");
                }

                testDataSourceInitted = true;
            }
        }
    }

    @Override
    public synchronized void close() throws Exception {
        if(targetDataSources != null){
            for(Object object : targetDataSources.values()){
                try {
                    if (object instanceof BasicDataSource) {
                        ((BasicDataSource) object).close();
                    } else if (object instanceof AutoCloseable) {
                        ((AutoCloseable) object).close();
                    }
                }catch(Exception e){
                    // ignore
                }
            }
        }
    }

    public BasicDataSource getOriginalDataSource() {
        return originalDataSource;
    }

    public BasicDataSource getShadeDataSource() {
        return shadeDataSource;
    }

}
