package cn.com.duibaboot.ext.autoconfigure.core.rpc;

import java.lang.reflect.Method;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

/**
 * RpcContext，用于存放RPC调用时的一些上下文信息，由于使用了TransmittableThreadLocal，故上下文信息可以传递到当前线程的子线程。
 */
public class RpcContext implements Cloneable {
    //TransmittableThreadLocal?
    private static final ThreadLocal<RpcContext> LOCAL = new ThreadLocal<>();
    /**
     * 当前rpc调用的方法
     */
    private Method method;
    /**
     * 当前rpc method调用时使用的参数
     */
    private Object[] invokeArgs;

    /**
     * 调用的来源服务名字
     */
    private String sourceServiceId;

    /**
     * 调用的目标服务名字
     */
    private String targetServiceId;

    /**
     * 当前rpc调用的本地地址，格式: ip:port，只有调用方有此数据(使用AtomicReference包装，让hystrix线程的修改可以让父线程可见)
     */
    private AtomicReference<String> localAddr = new AtomicReference<>();

    /**
     * 当前rpc调用的远程地址，格式: ip:port，只有调用方有此数据(使用AtomicReference包装，让hystrix线程的修改可以让父线程可见)
     */
    private AtomicReference<String> remoteAddr = new AtomicReference<>();

//    /**
//     * 当前调用RPC使用的参数
//     */
//    private Object[] arguments;
    /**
     * 当前存放的信息
     */
    private final Map<String, String> attachments = new HashMap();
    private final Map<String, Object> values = new HashMap();

    public static RpcContext getContext() {
        RpcContext c = LOCAL.get();
        if(c == null){
            c = new RpcContext();
            LOCAL.set(c);
        }

        return c;
    }

    public static boolean hasContext() {
        return LOCAL.get() != null;
    }

    public static void setContext(RpcContext rpcContext) {
        LOCAL.set(rpcContext);
    }

    public static void removeContext() {
        LOCAL.remove();
    }

    protected RpcContext() {
    }

    /**
     * 这个方法会在hystrix切换线程时被调用，以传递到下一个线程
     * @return
     */
    @Override
    public RpcContext clone(){ //NOSONAR
        RpcContext nr = new RpcContext();
        nr.getAttachments().putAll(this.getAttachments());
        nr.get().putAll(this.get());
        nr.setMethod(this.getMethod());
        nr.setInvokeArgs(this.getInvokeArgs());
        nr.setSourceServiceId(this.getSourceServiceId());
        nr.setTargetServiceId(this.getTargetServiceId());

        //以下两个属性是AtomicReference包装过的，为了让一个线程的修改对另一个线程可见，不能使用get/set方法
        nr.localAddr = this.localAddr;
        nr.remoteAddr = this.remoteAddr;

        return nr;
    }

//    public Object[] getArguments() {
//        return this.arguments;
//    }
//
//    public void setArguments(Object[] arguments) {
//        this.arguments = arguments;
//    }

    public String getAttachment(String key) {
        return this.attachments.get(key);
    }

    public RpcContext setAttachment(String key, String value) {
        if (value == null) {
            this.attachments.remove(key);
        } else {
            this.attachments.put(key, value);
        }

        return this;
    }

    public RpcContext removeAttachment(String key) {
        this.attachments.remove(key);
        return this;
    }

    public Map<String, String> getAttachments() {
        return this.attachments;
    }

    public RpcContext setAttachments(Map<String, String> attachment) {
        this.attachments.clear();
        if (attachment != null && attachment.size() > 0) {
            this.attachments.putAll(attachment);
        }

        return this;
    }

    public void clearAttachments() {
        this.attachments.clear();
    }

    public Map<String, Object> get() {
        return this.values;
    }

    public RpcContext set(String key, Object value) {
        if (value == null) {
            this.values.remove(key);
        } else {
            this.values.put(key, value);
        }

        return this;
    }

    public RpcContext remove(String key) {
        this.values.remove(key);
        return this;
    }

    public Object get(String key) {
        return this.values.get(key);
    }

    public Method getMethod() {
        return method;
    }

    public void setMethod(Method method) {
        this.method = method;
    }

    public Object[] getInvokeArgs() {
        return invokeArgs;
    }

    public void setInvokeArgs(Object[] invokeArgs) {
        this.invokeArgs = invokeArgs;
    }

    public String getLocalAddr() {
        return localAddr.get();
    }

    public void setLocalAddr(String localAddr) {
        this.localAddr.set(localAddr);
    }

    public String getRemoteAddr() {
        return remoteAddr.get();
    }

    public void setRemoteAddr(String remoteAddr) {
        this.remoteAddr.set(remoteAddr);
    }

    public String getSourceServiceId() {
        return sourceServiceId;
    }

    public void setSourceServiceId(String sourceServiceId) {
        this.sourceServiceId = sourceServiceId;
    }

    public String getTargetServiceId() {
        return targetServiceId;
    }

    public void setTargetServiceId(String targetServiceId) {
        this.targetServiceId = targetServiceId;
    }
}
