package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayConstants;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.GuavaCacheFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * guava缓存的方法录制
 * Created by guoyanfei .
 * 2019-04-15 .
 */
@Slf4j
public class RecordGuavaCacheMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以录制
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canRecord(Method method, Object[] allArguments) {
        if (!FlowReplayTrace.isTraced()) {
            return false;
        }
        if (!FlowReplayConstants.GUAVA_CACHE_AOP_METHOD.contains(method.getName())) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        if (!canRecord(method, allArguments)) {
            return;
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return;
        }
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(obj, method.getName(), allArguments);
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        if (!canRecord(method, allArguments)) {
            return ret;
        }
        if (!IgnoreSubInvokesContext.isInstMarked(obj, method.getName(), allArguments)) {
            return ret;
        }
        try {
            GuavaCacheFlowReplaySpan span = GuavaCacheFlowReplaySpan.createSpan(method, allArguments, argumentsTypes, ret);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
        } catch (Throwable t) {
            log.error("GuavaCache_录制异常", t);
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // 如果捕捉到异常，并且正在录制中，那么不录这个用例
        if (FlowReplayTrace.isTraced()) {
            FlowReplayTrace.remove();
        }
    }


}
