package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.RecordDetailContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.endpoint.ReplayConfigDto;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * 实例的回放上下文
 * Created by guoyanfei .
 * 2019-02-21 .
 */
@Slf4j
public final class ReplayContext {

    private static final int TRACE_QUEUE_SIZE = 500;
    private static final int RESULT_QUEUE_SIZE = 500;

    /**
     * 报告id
     */
    private final Long reportId;

    /**
     * 用例集文件地址
     */
    private final String usecaseSetUrl;

    private final Integer replayThreadPoolSize;

    private volatile Long startTime;

    private volatile Long endTime;

    /**
     * 回放的trace队列
     */
    private LinkedBlockingQueue<FlowReplayTrace> traceQueue;

    /**
     * trace是否加载结束
     */
    private volatile boolean tracesLoadFinished;

    /**
     * 回放结果的队列
     */
    private LinkedBlockingQueue<ReplayTraceResult> resultQueue;

    /**
     * 回放操作结束
     * 但是结果计算可能并未结束
     */
    private volatile boolean replayersDone;

    private volatile ReplayResult replayResult;

    /**
     * 构建回放详情所需的一些上下文内容（实际也是一次录制）
     */
    private final RecordDetailContext replayDetailContext;

    public ReplayContext(ReplayConfigDto replayConfig) {
        if (replayConfig == null || !replayConfig.isLegal()) {
            throw new FlowReplayException("回放配置为空or参数不合法");
        }
        this.reportId = replayConfig.getReportId();
        this.usecaseSetUrl = replayConfig.getUsecaseSetUrl();
        this.traceQueue = new LinkedBlockingQueue<>(TRACE_QUEUE_SIZE);
        this.resultQueue = new LinkedBlockingQueue<>(RESULT_QUEUE_SIZE);
        this.tracesLoadFinished = false;
        this.replayersDone = false;
        this.startTime = null;
        this.endTime = null;
        this.replayResult = null;
        this.replayThreadPoolSize = replayConfig.getReplayThreadPoolSize();
        this.replayDetailContext = new RecordDetailContext(replayConfig.getStackFramesPrefixWhitelist());
    }

    public RecordDetailContext getReplayDetailContext() {
        return replayDetailContext;
    }

    /**
     * 开始回放
     */
    public void start() {
        this.startTime = System.currentTimeMillis();
    }

    /**
     * 结束回放
     */
    public void end() {
        this.endTime = System.currentTimeMillis();
        this.traceQueue.clear();
        this.resultQueue.clear();
    }

    /**
     * 把新的trace添加到trace队列的末尾
     * @param trace
     * @throws InterruptedException
     */
    public void putTrace(FlowReplayTrace trace) throws InterruptedException {
        this.traceQueue.put(trace);
    }

    /**
     * 从trace队列中获取并删除头部的trace
     * @return
     * @throws InterruptedException
     */
    public FlowReplayTrace pollTrace() throws InterruptedException {
        return this.traceQueue.poll(2, TimeUnit.SECONDS);
    }

    /**
     * trace队列是不是空的
     * @return
     */
    public boolean isTraceQueueEmpty() {
        return this.traceQueue.size() < 1;
    }

    /**
     * trace队列标记为加载结束
     */
    public void tracesLoadFinish() {
        this.tracesLoadFinished = true;
        log.debug("引流回归_用例集加载结束_reportId_{}", reportId);
    }

    /**
     * trace队列加载是否完成
     * @return
     */
    public boolean isTracesLoadFinished() {
        return this.tracesLoadFinished;
    }

    /**
     * 把新的result添加到result队列的末尾
     * @param result
     * @throws InterruptedException
     */
    public void putResult(ReplayTraceResult result) throws InterruptedException {
        this.resultQueue.put(result);
    }

    /**
     * 从result队列中获取并删除头部的result
     * @return
     * @throws InterruptedException
     */
    public ReplayTraceResult pollResult() throws InterruptedException {
        return this.resultQueue.poll(2, TimeUnit.SECONDS);
    }

    /**
     * result队列是不是空的
     * @return
     */
    public boolean isResultQueueEmpty() {
        return this.resultQueue.size() < 1;
    }

    /**
     * result队列标记为添加完成
     */
    public void replayersDone() {
        this.replayersDone = true;
        log.debug("引流回归_回放器回放结束_reportId_{}", reportId);
    }

    /**
     * result队列添加是否完成
     * @return
     */
    public boolean isReplayersDone() {
        return this.replayersDone;
    }

    public Long getReportId() {
        return reportId;
    }

    public String getUsecaseSetUrl() {
        return usecaseSetUrl;
    }

    public Long getStartTime() {
        return startTime;
    }

    public Long getEndTime() {
        return endTime;
    }

    public ReplayResult getReplayResult() {
        return replayResult;
    }

    public void setReplayResult(ReplayResult replayResult) {
        this.replayResult = replayResult;
    }

    public Integer getReplayThreadPoolSize() {
        return replayThreadPoolSize;
    }
}
