package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.CaffeineCacheFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.SpanType;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.esotericsoftware.kryo.KryoException;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.github.benmanes.caffeine.cache.Policy;
import com.github.benmanes.caffeine.cache.stats.CacheStats;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;

import javax.annotation.CheckForNull;
import javax.annotation.Nonnull;
import java.lang.reflect.Method;
import java.util.Map;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Function;

/**
 * caffeine缓存的方法回放
 * Created by guoyanfei .
 * 2019-04-15 .
 */
@Slf4j
public class ReplayCaffeineCacheMethodInterceptor implements InstanceMethodsAroundInterceptor {

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        // do nothing
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        String methodName = method.getName();
        Assert.state("build".equals(methodName) || "buildAsync".equals(methodName), "method name must be 'build', will not be here");
        int parameterCount = method.getParameterCount();

        if ("buildAsync".equals(methodName) && parameterCount == 1) {//Caffeine.buildAsync方法
            // TODO 后续该方法也需要增强，暂时先不
            return ret;
        } else if ("build".equals(methodName) && parameterCount == 0) {//Caffeine.build无参方法
            Cache cache = (Cache) ret;
            return new ReplayCaffeineCacheMethodInterceptor.ReplayLocalCache(cache);
        } else if ("build".equals(methodName) && parameterCount == 1) {//Caffeine.build带CacheLoader的方法
            LoadingCache cache = (LoadingCache) ret;
            return new ReplayCaffeineCacheMethodInterceptor.ReplayLocalLoadingCache(cache);
        } else {
            throw new IllegalStateException("will never be here");
        }
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        //do nothing
    }

    static class ReplayLocalCache<K, V> implements Cache<K, V> {

        private Cache<K, V> cache;

        private ReplayLocalCache(Cache<K, V> cache) {
            this.cache = cache;
        }

        protected Cache<K, V> getCache() {
            return cache;
        }

        @CheckForNull
        @Override
        public V getIfPresent(@Nonnull Object key) {
            return executeReplay("getIfPresent", key, () -> getCache().getIfPresent(key));
        }

        @CheckForNull
        @Override
        public V get(@Nonnull K key, @Nonnull Function<? super K, ? extends V> mappingFunction) {
            return executeReplay("get", key, () -> getCache().get(key, mappingFunction));
        }

        @Nonnull
        @Override
        public Map<K, V> getAllPresent(@Nonnull Iterable<?> keys) {
            return executeReplay("getAllPresent", keys, () -> getCache().getAllPresent(keys));
        }

        @Override
        public void put(@Nonnull K key, @Nonnull V value) {
            getCache().put(key, value);
        }

        @Override
        public void putAll(@Nonnull Map<? extends K, ? extends V> map) {
            getCache().putAll(map);
        }

        @Override
        public void invalidate(@Nonnull Object key) {
            getCache().invalidate(key);
        }

        @Override
        public void invalidateAll(@Nonnull Iterable<?> keys) {
            getCache().invalidateAll(keys);
        }

        @Override
        public void invalidateAll() {
            getCache().invalidateAll();
        }

        @Override
        public long estimatedSize() {
            return getCache().estimatedSize();
        }

        @Nonnull
        @Override
        public CacheStats stats() {
            return getCache().stats();
        }

        @Nonnull
        @Override
        public ConcurrentMap<K, V> asMap() {
            return getCache().asMap();
        }

        @Override
        public void cleanUp() {
            getCache().cleanUp();
        }

        @Nonnull
        @Override
        public Policy<K, V> policy() {
            return getCache().policy();
        }

    }

    static class ReplayLocalLoadingCache<K, V> extends ReplayLocalCache<K, V> implements LoadingCache<K, V> {

        private ReplayLocalLoadingCache(Cache<K, V> cache) {
            super(cache);
        }

        protected LoadingCache<K, V> getCache() {
            return (LoadingCache) super.getCache();
        }

        @CheckForNull
        @Override
        public V get(@Nonnull K key) {
            return executeReplay("get", key, () -> getCache().get(key));
        }

        @Nonnull
        @Override
        public Map<K, V> getAll(@Nonnull Iterable<? extends K> keys) {
            return executeReplay("getAll", keys, () -> getCache().getAll(keys));
        }

        @Override
        public void refresh(@Nonnull K key) {
            getCache().refresh(key);
        }
    }

    /**
     * 回调接口
     *
     * @param <R>
     */
    private interface Callback<R> {

        R invoke();
    }

    private static <T> T executeReplay(String methodName, Object key, Callback<T> callback) {
        if (!canReplay(key)) {
            return callback.invoke();
        }

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回放的span，用于回放详情的构建
        CaffeineCacheFlowReplaySpan replayDetailSpan = createReplayDetailSpan(methodName, key, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.CAFFEINE_CACHE != span.getSpanType()) {
            String expert = SpanType.CAFFEINE_CACHE.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        CaffeineCacheFlowReplaySpan caffeineSpan = (CaffeineCacheFlowReplaySpan) span;

        if (!caffeineSpan.getMethodName().equals(methodName)) {
            String expert = caffeineSpan.getMethodName();
            String actual = methodName;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_502, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        Class<?>[] argumentsTypes = new Class[] { key.getClass() };
        if (!FlowReplayUtils.isArgumentsTypesEqual(caffeineSpan.getParameterTypes(), CaffeineCacheFlowReplaySpan.customizeArgumentsTypes(methodName, argumentsTypes))) {
            String expert = FlowReplayUtils.stringArrayToString(caffeineSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(argumentsTypes);
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_500, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        Object[] allArguments = new Object[] { key };
        try {
            if (!FlowReplayUtils.isArgumentsEqual(caffeineSpan.getParameterValues(), CaffeineCacheFlowReplaySpan.customizeParameterValues(methodName, allArguments))) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_501);
                throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
            }
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_503, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            T mockRet = (T) caffeineSpan.getReturnValue();
            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回放详情用的span中
            replayDetailSpan.setRet(mockRet);
            return mockRet;
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_504, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    private static CaffeineCacheFlowReplaySpan createReplayDetailSpan(String methodName, Object key, FlowReplaySpan span) {
        Object[] allArguments = new Object[] { key };
        Class<?>[] argumentsTypes = new Class[] { key.getClass() };
        CaffeineCacheFlowReplaySpan replayDetailSpan = CaffeineCacheFlowReplaySpan.createSpan(methodName, allArguments, argumentsTypes, null);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }

    /**
     * 可以回放
     *
     * @param method
     * @param allArguments
     * @return
     */
    private static boolean canReplay(Object arg) {
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }
        if (arg instanceof Iterable<?>) {
            for (Object o : (Iterable<?>) arg) {
                if (o == null) {
                    continue;
                }
                if (FlowReplayConstants.CANNOT_DESERIALIZE_CLASSES.contains(o.getClass().getName())) {
                    return false;
                }
            }
        }
        if (FlowReplayConstants.CANNOT_DESERIALIZE_CLASSES.contains(arg.getClass().getName())) {
            return false;
        }
        return true;
    }

}
