package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayErrorMsgTypeEnum;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.MybatisFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.SpanType;
import com.esotericsoftware.kryo.KryoException;
import org.apache.ibatis.cache.CacheKey;
import org.apache.ibatis.executor.Executor;
import org.apache.ibatis.mapping.BoundSql;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.plugin.*;
import org.apache.ibatis.session.ResultHandler;
import org.apache.ibatis.session.RowBounds;

import java.util.Objects;
import java.util.Properties;

/**
 * 对MyBatis进行拦截，回归
 */
@Intercepts({
        @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class }),
        @Signature(method = "query", type = Executor.class, args = { MappedStatement.class, Object.class, RowBounds.class, ResultHandler.class, CacheKey.class, BoundSql.class }),
        @Signature(method = "update", type = Executor.class, args = { MappedStatement.class, Object.class }) })
public class ReplayMybatisPlugin implements Interceptor {

    @Override
    public Object intercept(Invocation invocation) throws Throwable {
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return invocation.proceed();
        }

        MappedStatement ms = (MappedStatement) invocation.getArgs()[0];
        Object parameter = invocation.getArgs()[1];

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回放的span，用于回放详情的构建
        MybatisFlowReplaySpan replayDetailSpan = createReplayDetailSpan(ms.getId(), parameter, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.MYBATIS != span.getSpanType()) {
            String expert = SpanType.MYBATIS.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        MybatisFlowReplaySpan mybatisSpan = (MybatisFlowReplaySpan) span;

        // 方法是否相同
        if (!mybatisSpan.getMappedStatementId().equals(ms.getId())) {
            String expert = mybatisSpan.getMappedStatementId();
            String actual = ms.getId();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1021, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        // 参数类型是否相同
        String recordParameterType = mybatisSpan.getParameterType();
        String replayParameterType = (parameter != null ? parameter.getClass().getName() : null);
        if (!Objects.equals(recordParameterType, replayParameterType)) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1022, recordParameterType, replayParameterType);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            // 参数值是否相同
            if (!FlowReplayUtils.isObjectEqual(mybatisSpan.getParameterValue(), parameter)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1023);
                throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
            }
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1025, e);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            // 都相同mock返回值
            Object mockRet = mybatisSpan.getReturnValue();

            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回放详情用的span中
            replayDetailSpan.setRet(mockRet);
            return mockRet;
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_1026, e);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

    }

    private MybatisFlowReplaySpan createReplayDetailSpan(String mappedStatementId, Object parameter, FlowReplaySpan span) {
        MybatisFlowReplaySpan replayDetailSpan = MybatisFlowReplaySpan.createSpan(mappedStatementId, parameter);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }

    @Override
    public Object plugin(Object target) {
        return Plugin.wrap(target, this);
    }

    @Override
    public void setProperties(Properties arg0) {
        // do nothing
    }

}
