package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.RandomFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.SpanType;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.StaticMethodsAroundInterceptor;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * 随机数的方法回放
 * Created by guoyanfei .
 * 2019-05-17 .
 */
@Slf4j
public class ReplayRandomMethodInterceptor implements StaticMethodsAroundInterceptor {

    /**
     * 可以回放
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Class clazz, Method method, Object[] allArguments, Class<?>[] parameterTypes, MethodInterceptResult result) {
        if (!canReplay(method, allArguments)) {
            return;
        }

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回放的span，用于回放详情的构建
        RandomFlowReplaySpan replayDetailSpan = createReplayDetailSpan(method, allArguments, parameterTypes, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.RANDOM != span.getSpanType()) {
            String expert = SpanType.RANDOM.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        RandomFlowReplaySpan randomSpan = (RandomFlowReplaySpan) span;

        if (!FlowReplayUtils.isMethodEqual(randomSpan.getMethodName(), method)) {
            String expert = randomSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_602, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!FlowReplayUtils.isArgumentsTypesEqual(randomSpan.getParameterTypes(), parameterTypes)) {
            String expert = FlowReplayUtils.stringArrayToString(randomSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(parameterTypes);
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_600, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            if (!FlowReplayUtils.isArgumentsEqual(randomSpan.getParameterValues(), allArguments)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_601);
                throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
            }
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_603, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            Object mockRet = randomSpan.getReturnValue();

            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回放详情用的span中
            replayDetailSpan.setReturnValue(mockRet);

            result.defineReturnValue(mockRet);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_604, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    private RandomFlowReplaySpan createReplayDetailSpan(Method method, Object[] allArguments, Class<?>[] parameterTypes, FlowReplaySpan span) {
        RandomFlowReplaySpan replayDetailSpan = RandomFlowReplaySpan.createSpan(method, allArguments, parameterTypes, null);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }

    @Override
    public Object afterMethod(Class clazz, Method method, Object[] allArguments, Class<?>[] parameterTypes, Object ret) {
        return ret;
    }

    @Override
    public void handleMethodException(Class clazz, Method method, Object[] allArguments, Class<?>[] parameterTypes, Throwable t) {
        // do nothing
    }
}
