package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.RedisFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.SpanType;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * RedisLock 回放
 * Created by guoyanfei .
 * 2019-06-20 .
 */
@Slf4j
public class ReplayRedisLockMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以回放
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }
        if (!"unlock".equals(method.getName())) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        if (!canReplay(method, allArguments)) {
            return;
        }
        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回放的span，用于回放详情的构建
        RedisFlowReplaySpan replayDetailSpan = createReplayDetailSpan(method, allArguments, argumentsTypes, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.REDIS != span.getSpanType()) {
            String expert = SpanType.REDIS.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        RedisFlowReplaySpan redisSpan = (RedisFlowReplaySpan) span;

        if (!FlowReplayUtils.isMethodEqual(redisSpan.getMethodName(), method)) {
            String expert = redisSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_102, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            Object mockRet = redisSpan.getReturnValue();
            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回放详情用的span中
            replayDetailSpan.setReturnValue(mockRet);

            result.defineReturnValue(mockRet);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_104, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    private RedisFlowReplaySpan createReplayDetailSpan(Method method, Object[] allArguments, Class<?>[] argumentsTypes, FlowReplaySpan span) {
        RedisFlowReplaySpan replayDetailSpan = RedisFlowReplaySpan.createSpan(method, allArguments, argumentsTypes, null);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // do nothing
    }
}
