package cn.com.duibaboot.ext.autoconfigure.initserver;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.JarVersionUtils;
import cn.com.duiba.boot.utils.MainApplicationContextHolder;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duiba.wolf.utils.ConcurrentUtils;
import cn.com.duibaboot.ext.autoconfigure.core.AsyncSpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.data.redis.RedisProperties;
import cn.com.duibaboot.ext.autoconfigure.web.BootMonitorCheckFilter;
import com.alibaba.dubbo.rpc.service.EchoService;
import com.alibaba.fastjson.JSON;
import com.google.common.base.Joiner;
import com.mysql.jdbc.Driver;
import org.apache.commons.dbcp2.BasicDataSource;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hbase.client.Result;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.ApplicationRunner;
import org.springframework.boot.SpringBootVersion;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.boot.web.servlet.ServletRegistrationBean;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.dao.DataAccessException;
import org.springframework.data.elasticsearch.core.ElasticsearchOperations;
import org.springframework.data.hadoop.hbase.HbaseTemplate;
import org.springframework.data.hadoop.hbase.RowMapper;
import org.springframework.data.mongodb.core.MongoOperations;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.jdbc.CannotGetJdbcConnectionException;
import redis.clients.jedis.Jedis;

import javax.annotation.PostConstruct;
import javax.servlet.Filter;
import javax.sql.DataSource;
import java.io.IOException;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 应用启动时初始化，防止第一次请求执行慢
 */
@Configuration
public class InitServerAutoConfiguration {
	private static Logger logger = LoggerFactory.getLogger(InitServerAutoConfiguration.class);

    @Value("${server.port}")
    private int port;

	@Autowired
	private ConfigurableApplicationContext applicationContext;

	@PostConstruct
	public void init() {
		MainApplicationContextHolder._setApplicationContext(applicationContext);

		RedisProperties rp = new RedisProperties();
		try {
			BeanUtils.copy(rp, RedisProperties.class);
		}catch(Exception e){
			throw new IllegalStateException("尝试调用BeanUtils.copy方法时失败，请检查cglib库是否存在jar包冲突，只保留cglib:cglib依赖即可", e);
		}

		//检测wolf版本
		String wolfVersion = JarVersionUtils.getJarVersion(ConcurrentUtils.class);
		if(StringUtils.isBlank(wolfVersion) || !JarVersionUtils.isJarVersionEqualOrGreaterThan(wolfVersion, "2.2.23")){
			throw new IllegalStateException("检测到wolf版本过低，请去掉对('cn.com.duiba:wolf')的依赖，由spring-boot-ext来统一管理wolf的版本");
		}

		checkInetAddress();
		checkSpringDataRedisVersion();
		checkSpringBootVersion();
		checkMysqlVersion();
		checkFastjsonVersion();
	}

	//检查 InetAddress 获取hostName是否超时
	private void checkInetAddress(){
		String[] profiles = applicationContext.getEnvironment().getActiveProfiles();
		for(String env : profiles){
			if(SpringEnvironmentUtils.DEV.equals(env)){
				long s = System.currentTimeMillis();
				try {
					InetAddress.getLocalHost().getHostAddress();
				} catch (UnknownHostException e) {
					//ignore
				}
				long e = System.currentTimeMillis();
				if( e -s > 500){
					throw new IllegalStateException("检测到获取host超时，请配置hosts，参考：http://cf.dui88.com/pages/viewpage.action?pageId=33995153");
				}
			}
		}
	}

	//检查spring-data-redis版本号是否过低
	private void checkSpringDataRedisVersion(){
		boolean isRedisClassExists = false;
		try {
			Class.forName("org.springframework.data.redis.connection.RedisConnection");
			isRedisClassExists = true;
		} catch (ClassNotFoundException e) {
			//Ignore
		}
		if(isRedisClassExists){
			try {
				Class.forName("org.springframework.data.redis.core.types.Expiration");
			} catch (ClassNotFoundException e) {
				throw new IllegalStateException("检测到spring-data-redis版本过低，请去掉对spring-data-redis的版本号指定，由spring-boot-ext来统一管理spring-data-redis的版本");
			}
		}
	}

	//检查fastjson版本号是否过低
	private void checkFastjsonVersion(){
		boolean isFastJsonClassExists = false;
		try {
			Class.forName("com.alibaba.fastjson.JSON");
			isFastJsonClassExists = true;
		} catch (ClassNotFoundException e) {
			//Ignore
		}
		if(isFastJsonClassExists) {
			if (!JSON.VERSION.contains(".sec") && !JarVersionUtils.isJarVersionEqualOrGreaterThan(JSON.VERSION, "1.2.60")) {
				throw new IllegalStateException("检测到fastjson版本过低，建议去掉对fastjson的版本号指定，由spring-boot-ext来统一管理fastjson的版本");
			}
		}
	}

	//检查spring-boot是否使用了有问题的版本号
	private void checkSpringBootVersion(){
		if("1.5.19.RELEASE".equals(SpringBootVersion.getVersion())) {
			throw new IllegalStateException("检测到当前项目使用的spring-boot版本为1.5.19.RELEASE，该版本引入的jedis版本存在严重问题，会导致redis连接泄露，请勿使用该版本!");
		}
		Class<?> jedisClass = null;
		try {
			jedisClass = Class.forName("redis.clients.jedis.Jedis");
		} catch (ClassNotFoundException e) {
			//Ignore
		}

		if(jedisClass != null) {
			if("2.9.1".equals(JarVersionUtils.getJedisVersion())){
				throw new IllegalStateException("检测到当前项目使用的jedis版本为2.9.1，该jedis版本存在严重问题，会导致redis连接泄露，请勿使用该版本!");
			}
		}
	}

	//检查mysql版本号
	private void checkMysqlVersion(){
		Class<?> mysqlDriverClass = null;
		try {
			mysqlDriverClass = Class.forName("com.mysql.jdbc.Driver");
		} catch (ClassNotFoundException e) {
			//Ignore
		}

		if(mysqlDriverClass != null) {
			Package pkg = Driver.class.getPackage();
			String version =  (pkg != null) ? pkg.getImplementationVersion() : null;

			if(!StringUtils.isBlank(version)
					&& !JarVersionUtils.isJarVersionEqualOrGreaterThan(version, "5.1.45")){
				throw new IllegalStateException("检测到当前项目使用的 mysql-connector-java 版本为"+version+"，请升级到5.1.45或以上版本(建议去掉对mysql-connector-java的版本号指定，由spring-boot-ext来统一管理mysql-connector-java的版本)!");
			}
		}
	}

	/**
	 * 用于发送MainContextRefreshedEvent事件
	 * @return
	 */
	@Bean
	public ApplicationRunner mainContextRefreshedEventPublishApplicationRunner(){
		return new MainContextRefreshedEventPublishApplicationRunner();
	}

	/**
	 * 如果检测到用户设置了filter，给出警告信息
	 */
	@EventListener(MainContextRefreshedEvent.class)
	public void onMainContextRefreshed(){
        Map<String, Filter> filtersMap = null;
        Map<String, FilterRegistrationBean> filterRegistrationBeanMap = null;
	    try {
            filtersMap = applicationContext.getBeansOfType(Filter.class);
        }catch(BeansException e){
            logger.warn("", e);
        }
	    try {
            filterRegistrationBeanMap = applicationContext.getBeansOfType(FilterRegistrationBean.class);
        }catch(BeansException e){
	        logger.warn("", e);
        }
        List<Filter> allFilters = new ArrayList<>();
        if(filtersMap != null){
            allFilters.addAll(filtersMap.values());
        }
		if(filterRegistrationBeanMap != null && !filterRegistrationBeanMap.isEmpty()){
			for(FilterRegistrationBean b : filterRegistrationBeanMap.values()){
				allFilters.add(b.getFilter());
			}
		}

		List<String> unexpectedFilterClassNames = new ArrayList<>();

		for(Filter filter: allFilters){
			String className = filter.getClass().getName();
			if(!className.startsWith("org.springframework.")
					&& !className.startsWith("cn.com.duibaboot.")
					&& !className.startsWith("com.sun.jersey")){
				unexpectedFilterClassNames.add(className);
			}
		}

		if(!unexpectedFilterClassNames.isEmpty()
				&& logger.isWarnEnabled()){
			logger.warn("检测到您的项目中使用了Filter，请尽量使用HandlerInterceptor来代替Filter,以避免线上压测出现问题，避免拦截spring-boot-starter-actuator的endpoint;您使用到的Filter类有如下这些: {}", Joiner.on(",").join(unexpectedFilterClassNames));
		}
	}

	/**
	 * 用于管理所有的SpecifiedBeanPostProcessor，对他们进行调用
	 */
	@Bean
	public static BeanPostProcessorOfSpecified specifiedBeanPostProcessor(){
		return new BeanPostProcessorOfSpecified();
	}

	/**
	 * 用于管理所有的AsyncSpecifiedBeanPostProcessor，对他们进行调用,这个Listener会阻塞住spring流程，确保异步初始化全部执行完成再继续
	 */
	@Bean
	public ApplicationListenerForAsyncSpecified asyncSpecifiedBeanPostProcessorConfigurer(BeanPostProcessorOfSpecified specifiedBeanPostProcessor){
		//找到所有SpecifiedBeanPostProcessor实例并排序，然后在ApplicationListener中按顺序处理
		return new ApplicationListenerForAsyncSpecified(specifiedBeanPostProcessor.getAsyncProcessor2BeansMap());
	}

	/**
	 * 初始化Servlet
	 */
	@Configuration
	@ConditionalOnWebApplication
	@ConditionalOnClass({ServletRegistrationBean.class})
	public static class SpringServletInitConfiguration{
		
		@Bean
		public static SpecifiedBeanPostProcessor webPostProcessorConfigurer(){
			return new SpecifiedBeanPostProcessor<ServletRegistrationBean>() {

				@Override
				public int getOrder() {
					return 0;
				}

				@Override
				public Class<ServletRegistrationBean> getBeanType() {
					return ServletRegistrationBean.class;
				}

				@Override
				public Object postProcessBeforeInitialization(ServletRegistrationBean bean, String beanName) throws BeansException {
					return bean;
				}

				@Override
				public Object postProcessAfterInitialization(ServletRegistrationBean bean, String beanName) throws BeansException {
					//启动时初始化 dispatcherServlet
					if(bean.getServletName().equalsIgnoreCase("dispatcherServlet")){
						bean.setLoadOnStartup(1);
					}
					return bean;
				}
			};
		}
	}

	/**
	 * 监听到MainContextRefreshedEvent后访问一次/monitor/check接口，做预热（第一次访问会比较慢）,访问完成后放开服务
	 */
	@EventListener(MainContextRefreshedEvent.class)
	@Order(Ordered.LOWEST_PRECEDENCE)//顺序最靠后，等其他地方处理MainContextRefreshedEvent时间完成后再使/monitor/check可用。（所有MainContextRefreshedEvent监听处理完毕后才会注册到eureka）
	public void onMainContextRefreshed(MainContextRefreshedEvent event) {
		try(
			CloseableHttpClient httpClient = HttpClientBuilder.create()
					.setDefaultRequestConfig(RequestConfig.custom().setConnectTimeout(200).setSocketTimeout(5000).setConnectionRequestTimeout(100).build())
					.setMaxConnPerRoute(1)
					.setMaxConnTotal(1)
					.setUserAgent("DuibaBootInit")
					.disableAutomaticRetries()//禁止重试
					.disableCookieManagement()
					.disableRedirectHandling()
					.evictExpiredConnections()//每隔10秒主动扫描并逐出超时的连接（超过keepAliveTimeout）
					.build()
		){
			HttpGet monitorCheckReq = new HttpGet("http://localhost:" + port + "/monitor/check");
			CloseableHttpResponse resp = httpClient.execute(monitorCheckReq);
			resp.close();
		} catch (IOException e) {
			logger.warn("init self failed", e);
		}

		//设置开始提供服务
		BootMonitorCheckFilter.setInService(true);
	}

	/**
	 * 并发初始化Dubbo
	 */
	@Configuration
	@ConditionalOnClass({EchoService.class})
	public static class DubboPostProcessorConfiguration{
		
		@Bean
		public AsyncSpecifiedBeanPostProcessor dubboPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<EchoService>() {

				@Override
				public Class<EchoService> getBeanType() {
					return EchoService.class;
				}

				@Override
				public void postProcessAfterInitialization(EchoService echoService, String beanName) {
					//dubbo回声检测，用于检测dubbo调用链路是否通畅 （即初始化dubbo）
					try {
						echoService.$echo("OK");
					} catch (Exception e) {
						logger.error(e.getMessage(), e);
					}
				}
			};
		}
	}

	/**
	 * 初始化jdbc
	 */
	@Configuration
	@ConditionalOnClass({CannotGetJdbcConnectionException.class, BasicDataSource.class})
	public static class DataSourcePostProcessorConfiguration{
		
		@Bean
		public AsyncSpecifiedBeanPostProcessor dataSourcePostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<DataSource>() {

				@Override
				public Class<DataSource> getBeanType() {
					return DataSource.class;
				}
				
				@Override
				public void postProcessAfterInitialization(DataSource dataSource, String beanName) {
					//初始化数据库连接池
					Connection conn = null;
					try {
						conn = dataSource.getConnection();
					} catch (SQLException e) {
						throw new CannotGetJdbcConnectionException("Could not get JDBC Connection", e);
					} finally{
						if(conn != null){
							try {
								conn.close();
							} catch (SQLException e) {}
						}
					}
				}
			};
		}
	}

	/**
	 * 初始化redis
	 */
	@Configuration
	@ConditionalOnClass({RedisTemplate.class})
	public static class RedisPostProcessorConfiguration{

		@Bean
		public AsyncSpecifiedBeanPostProcessor redisPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<RedisTemplate>() {

				@Override
				public Class<RedisTemplate> getBeanType() {
					return RedisTemplate.class;
				}

				@Override
				public void postProcessAfterInitialization(RedisTemplate redisTemplate, String beanName) {
					//调用redis初始化，由于第一次redis调用会比较慢（尤其是其中Cat监控会给redisTemplate加上proxy，由于第一次生成proxy类比较慢，大约需要500ms，刚启动时如果有大量redis调用，都需要500多ms），所以需要调用预热一次，加速启动完成后的访问速度
					try {
						redisTemplate.execute(new RedisCallback() {
							@Override
							public Object doInRedis(RedisConnection connection) throws DataAccessException {
								Jedis jedis = (Jedis) connection.getNativeConnection();
								return jedis.get("justForInitTest");
							}
						});
					}catch(Exception e){
						if(redisTemplate.getConnectionFactory() instanceof JedisConnectionFactory) {
							JedisConnectionFactory cf = (JedisConnectionFactory) redisTemplate.getConnectionFactory();
							logger.error("Init redisTemplate(beanName:{}) failed, currentHost:{}, port:{}", beanName, cf.getHostName(), cf.getPort(), e);
						}else{
							throw e;
						}
					}
				}
			};
		}
	}

	/**
	 * 初始化elasticSearch
	 */
	@Configuration
	@ConditionalOnClass({ElasticsearchOperations.class})
	public static class ElasticSearchPostProcessorConfiguration{

		@Bean
		public AsyncSpecifiedBeanPostProcessor elasticSearchPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<ElasticsearchOperations>() {

				@Override
				public Class<ElasticsearchOperations> getBeanType() {
					return ElasticsearchOperations.class;
				}

				@Override
				public void postProcessAfterInitialization(ElasticsearchOperations elasticsearchOperations, String beanName) {
					//调用elasticSearch初始化，由于第一次elasticSearch调用会比较慢（尤其是其中Cat监控会给elasticSearch加上proxy，由于第一次生成proxy类比较慢，大约需要500ms，刚启动时如果有大量elasticSearch调用，都需要500多ms），所以需要调用预热一次，加速启动完成后的访问速度
					elasticsearchOperations.indexExists("justForTest");
				}
			};
		}
	}

	/**
	 * 初始化MongoDb
	 */
	@Configuration
	@ConditionalOnClass({MongoOperations.class})
	public static class MongoDbPostProcessorConfiguration{

		@Bean
		public AsyncSpecifiedBeanPostProcessor mongoDbPostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<MongoOperations>() {

				@Override
				public Class<MongoOperations> getBeanType() {
					return MongoOperations.class;
				}

				@Override
				public void postProcessAfterInitialization(MongoOperations mongoOperations, String beanName) {
					//调用MongoDb初始化，由于第一次MongoDb调用会比较慢（尤其是其中Cat监控会给MongoDb加上proxy，由于第一次生成proxy类比较慢，大约需要500ms，刚启动时如果有大量MongoDb调用，都需要500多ms），所以需要调用预热一次，加速启动完成后的访问速度
					try {
						mongoOperations.findOne(Query.query(Criteria.where("id").is("1")), String.class,"test_collection");
					}catch(Exception e){
						logger.error(e.getMessage(), e);
					}
				}
			};
		}
	}

	/**
	 * 初始化Hbase
	 */
	@Configuration
	@ConditionalOnClass({HbaseTemplate.class})
	public static class HbasePostProcessorConfiguration{

		@Bean
		public AsyncSpecifiedBeanPostProcessor hbasePostProcessorConfigurer(){
			return new AsyncSpecifiedBeanPostProcessor<HbaseTemplate>() {

				@Override
				public Class<HbaseTemplate> getBeanType() {
					return HbaseTemplate.class;
				}

				@Override
				public void postProcessAfterInitialization(HbaseTemplate hbaseTemplate, String beanName) {
					//调用Hbase初始化，由于第一次Hbase调用会比较慢（尤其是其中Cat监控会给Hbase加上proxy，由于第一次生成proxy类比较慢，大约需要500ms，刚启动时如果有大量Hbase调用，都需要500多ms），所以需要调用预热一次，加速启动完成后的访问速度
					try {
						hbaseTemplate.get("hbase:meta", "1", new RowMapper<String>() {
							@Override
							public String mapRow(Result result, int rowNum) throws Exception {
								return null;
							}
						});
					}catch(Exception e){
						logger.error(e.getMessage(), e);
					}
				}
			};
		}
	}

	/**
	 * 写点注释,迁移到dubbo里（这个方法废弃）
	 */
//	@Configuration
//	@ConditionalOnClass({ReferenceBean.class})
//	public static class DubboPostFactoryProcessorConfiguration{
//
//		@Bean
//		public static BeanFactoryPostProcessor dubboBeanFactoryPostProcessorConfigurer(){
//			return new DubboBeanFactoryPostProcessor();
//		}
//	}
//
//	private static class DubboBeanFactoryPostProcessor implements Ordered, BeanFactoryPostProcessor {
//		@Override
//		public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
//			for(String beanName : beanFactory.getBeanDefinitionNames()){
//				BeanDefinition bd = beanFactory.getBeanDefinition(beanName);
//				if(bd.isSingleton() && !bd.isAbstract() && !bd.isLazyInit() && ReferenceBean.class.getName().equals(bd.getBeanClassName())){
//					try {
//						beanFactory.getBean(beanName);
//					}catch(NoSuchBeanDefinitionException e){
//						// Ignore
//					}
//				}
//			}
//		}
//
//		@Override
//		public int getOrder() {
//			return Ordered.LOWEST_PRECEDENCE;
//		}
//	}
}
