package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer;

import java.util.List;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import com.netflix.loadbalancer.AbstractLoadBalancer;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.LoadBalancerStats;
import com.netflix.loadbalancer.Server;
import com.netflix.loadbalancer.ServerStats;

/**
 * 加权最小连接数 负载均衡规则，加权最小连接调度在调度新连接时尽可能使服务器的已建立连接数和其权值成比例
 *
 * https://blog.csdn.net/starxu85/article/details/3230118
 *
 * 加权最小连接的算法流程:
 *
 * 假设有一组服务器S = {S0, S1, ..., Sn-1}，W(Si)表示服务器Si的权值，
 * C(Si)表示服务器Si的当前连接数。所有服务器当前连接数的总和为
 * CSUM = ΣC(Si)  (i=0, 1, .. , n-1)。当前的新连接请求会被发送服务器Sm，
 * 当且仅当服务器Sm满足以下条件
 * (C(Sm) / CSUM)/ W(Sm) = min { (C(Si) / CSUM) / W(Si)}  (i=0, 1, . , n-1)
 * 其中W(Si)不为零
 * 因为CSUM在这一轮查找中是个常数，所以判断条件可以简化为
 * C(Sm) / W(Sm) = min { C(Si) / W(Si)}  (i=0, 1, . , n-1)
 * 其中W(Si)不为零
 *
 * 因为除法所需的CPU周期比乘法多，且在Linux内核中不允许浮点除法，服务器的
 * 权值都大于零，所以判断条件C(Sm) / W(Sm) > C(Si) / W(Si) 可以进一步优化
 * 为C(Sm)*W(Si) > C(Si)* W(Sm)。同时保证服务器的权值为零时，服务器不被调
 * 度。所以，算法只要执行以下流程。
 *
 * for (m = 0; m < n; m++) {
 *  if (W(Sm) > 0) {
 *      for (i = m+1; i < n; i++) {
 *          if (C(Sm)*W(Si) > C(Si)*W(Sm))
 *              m = i;
 *      }
 *      return Sm;
 *  }
 * }
 * return NULL;
 *
 *
 * @author wenqi.huang
 */
public class WeightedLeastConnectionRule extends FilterBasedRule {

    public WeightedLeastConnectionRule(ILoadBalancer lb, List<RibbonServerListFilter> serverListFilters) {
        super(serverListFilters);
        setLoadBalancer(lb);
    }

    @Override
    protected Server chooseFromServers(List<Server> servers, ILoadBalancer lb, Object key) {
        int n = servers.size();
        if(n == 0){
            return null;
        }

        int weight[] = new int[servers.size()];
        int activeRequestCount[] = new int[servers.size()];
        AbstractLoadBalancer nlb = (AbstractLoadBalancer) lb;
        LoadBalancerStats stats = nlb.getLoadBalancerStats();
        for(int i=0;i<weight.length;i++){
            Server s = servers.get(i);
            weight[i] = getTimeBasedWeight(s);
            ServerStats stat = stats.getSingleServerStat(s);
            activeRequestCount[i] = stat.getActiveRequestsCount();
        }

        for (int m = 0; m < n; m++) {
            if (weight[m] <= 0) {
                continue;//权重为0的服务器不给予流量
            }

            int choosedIndex = m;
            for (int i = m + 1; i < n; i++) {
                if (activeRequestCount[choosedIndex] * weight[i]
                        > activeRequestCount[i] * weight[choosedIndex]) {
                    choosedIndex = i;
                }
            }
            return servers.get(choosedIndex);
        }

        return null;
    }

}
