package cn.com.duibaboot.ext.autoconfigure.datasource;

import cn.com.duibaboot.ext.autoconfigure.perftest.PerfTestRoutingDataSource;
import com.alibaba.fastjson.JSON;
import com.zaxxer.hikari.HikariDataSource;
import com.zaxxer.hikari.HikariPoolMXBean;
import lombok.extern.slf4j.Slf4j;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.apache.commons.dbcp2.BasicDataSource;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.ApplicationContext;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 用于数据库连接池的实时监控，每秒钟检测一次数据库源的bean，并保存到一个map中，用于endpoint调用
 * Created by guoyanfei .
 * 2018/6/13 .
 */
@Slf4j
public class DuibaDataSourceMonitor implements DisposableBean {

    private static final boolean HIKARI_CLASS_EXISTS;
    private static final boolean HIKARI_MONITOR_EXISTS;//HikariDataSource.getHikariPoolMXBean方法是否存在，如果不存在，代表hikari版本较低,无法监控连接池使用情况
    private static final boolean DBCP2_CLASS_EXISTS;

    static{
        boolean hikariClassExists = false;
        boolean hikariMonitorExists = false;
        boolean dbcp2ClassExists = false;
        try {
            Class.forName("com.zaxxer.hikari.HikariDataSource");
            hikariClassExists = true;
            HikariDataSource.class.getDeclaredMethod("getHikariPoolMXBean");
            hikariMonitorExists = true;
        } catch (ClassNotFoundException e) {
            //Ignore
        } catch (NoSuchMethodException e) {
            log.warn("请把hikari升级到最新版，否则无法监控连接池使用数据");
        }
        HIKARI_MONITOR_EXISTS = hikariMonitorExists;

        try {
            Class.forName("org.apache.commons.dbcp2.BasicDataSource");
            dbcp2ClassExists = true;
        } catch (ClassNotFoundException e) {
            //Ignore
        }

        HIKARI_CLASS_EXISTS = hikariClassExists;
        DBCP2_CLASS_EXISTS = dbcp2ClassExists;
    }

    private boolean initted = false;

    private Map<String, DataSource> dataSourceMap;

    private Thread duibaDataSourceMonitorThread;

    private static final long RETENTION_TIME = 1000L * 60 * 30;  // 30分钟

    private Map<String, Map<Long, MonitorObj>> secondMonitorMap = new ConcurrentHashMap<>();

    protected synchronized String getSecondMonitorJson() {
        return JSON.toJSONString(secondMonitorMap);
    }

    public synchronized void startMonitorThread(ApplicationContext context) {
        if (!initted) {
            initted = true;
            dataSourceMap = context.getBeansOfType(DataSource.class);

            if (dataSourceMap.isEmpty()) {
                return;
            }

            duibaDataSourceMonitorThread = new Thread(() -> {
                scanDataSources();
            },"duibaDataSourceMonitorThread");

            duibaDataSourceMonitorThread.start();
        }
    }

    private void scanDataSources(){
        while (true) {
            for (Map.Entry<String, DataSource> entry : dataSourceMap.entrySet()) {
                DataSource ds = entry.getValue();
                if(ds instanceof PerfTestRoutingDataSource){
                    PerfTestRoutingDataSource pds = (PerfTestRoutingDataSource)ds;
                    scanDataSource(entry.getKey(), pds.getOriginalDataSource());
                    scanDataSource(entry.getKey() + "_shade", pds.getShadeDataSource());
                } else {
                    scanDataSource(entry.getKey(), ds);
                }
            }

            try {
                Thread.sleep(1000);//每隔1S检查一次，记录数据库连接池使用情况
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
            if (Thread.currentThread().isInterrupted()) {
                break;
            }
        }
    }

    private synchronized void scanDataSource(String key, DataSource ds) {
        if (isValid(ds)) {
            MonitorObj currentMonitorObj = getCurrentMonitorObj(ds);

            if(currentMonitorObj == null){
                return;
            }

            warnIfTooFull(key, currentMonitorObj);

            Map<Long, MonitorObj> dbSecondMonitorMap = secondMonitorMap.computeIfAbsent(key, s -> {
                return new LinkedHashMap<Long, MonitorObj>() {
                    // put的时候，删除30分钟之前put的值
                    @Override
                    protected boolean removeEldestEntry(Map.Entry<Long, MonitorObj> eldest) {
                        return eldest.getKey() < (System.currentTimeMillis() - RETENTION_TIME);
                    }
                };
            });

            dbSecondMonitorMap.put(System.currentTimeMillis(), currentMonitorObj);
        }
    }

    /**
     * 数据库连接池检查报警,当连接池数目过满时打印报警日志
     * @param currentMonitorObj
     */
    private void warnIfTooFull(String key, MonitorObj currentMonitorObj){
        int numIdleConnections = currentMonitorObj.getNumAllocated() - currentMonitorObj.getNumActive();//池中有多少个空闲连接，它们可以被checkout

        if (currentMonitorObj.getNumActive() >= 3
                && currentMonitorObj.getNumAllocated() >= currentMonitorObj.getMaxTotal() - 3
                && currentMonitorObj.getNumActive() >= currentMonitorObj.getNumAllocated() - 3) {
            log.error("datasource [" + key + "]'s connectionPool is too full,max:" + currentMonitorObj.getMaxTotal() + ",busy:" + currentMonitorObj.getNumActive() + ",idle:" + numIdleConnections);
        }
    }

    private MonitorObj getCurrentMonitorObj(DataSource ds){
        if(DBCP2_CLASS_EXISTS && ds instanceof BasicDataSource){
            BasicDataSource ds1 = (BasicDataSource)ds;
            int numIdleConnections = ds1.getNumIdle(); // 池中有多少个空闲连接，它们可以被checkout
            int numActiveConnections = ds1.getNumActive();   // 池中被checkout的连接有多少个
            int maxTotal = ds1.getMaxTotal();//最大允许个数
            return new MonitorObj(numIdleConnections, numActiveConnections, maxTotal, -1);
        } else if(HIKARI_CLASS_EXISTS && ds instanceof HikariDataSource) {
            if(HIKARI_MONITOR_EXISTS) {
                HikariDataSource ds1 = (HikariDataSource) ds;
                HikariPoolMXBean poolMXBean = ds1.getHikariPoolMXBean();
                int numIdleConnections = poolMXBean.getIdleConnections(); // 池中有多少个空闲连接，它们可以被checkout
                int numActiveConnections = poolMXBean.getActiveConnections();   // 池中被checkout的连接有多少个
                int maxTotal = ((HikariDataSource) ds).getMaximumPoolSize();//最大允许个数
                int threadsAwaitingConnection = poolMXBean.getThreadsAwaitingConnection();
                return new MonitorObj(numIdleConnections, numActiveConnections, maxTotal, threadsAwaitingConnection);
            } else {
                return null;
            }
        } else {
            throw new IllegalStateException("[notifyme]will not be here");
        }
    }

    private boolean isValid(DataSource ds){
        if(ds != null){
            if(DBCP2_CLASS_EXISTS && ds instanceof BasicDataSource && !((BasicDataSource)ds).isClosed()){
                return true;
            } else if(HIKARI_CLASS_EXISTS && ds instanceof HikariDataSource && !((HikariDataSource)ds).isClosed()){
                return true;
            }
        }

        return false;
    }

    /**
     * 获取数据库连接池配置
     * @return
     */
    public List<DataSourceConfig> getDataSourceConfig() {
        List<DataSourceConfig> dataSourceConfigs = new ArrayList<>();
        for (Map.Entry<String, DataSource> entry : dataSourceMap.entrySet()) {
            DataSource ds = entry.getValue();
            if(ds instanceof PerfTestRoutingDataSource){
                PerfTestRoutingDataSource pds = (PerfTestRoutingDataSource)ds;
                scanDataSourceForConfig(entry.getKey(), pds.getOriginalDataSource(), dataSourceConfigs);
                scanDataSourceForConfig(entry.getKey() + "_shade", pds.getShadeDataSource(), dataSourceConfigs);
            } else {
                scanDataSourceForConfig(entry.getKey(), ds, dataSourceConfigs);
            }
        }

        return dataSourceConfigs;
    }

    private synchronized void scanDataSourceForConfig(String key, DataSource ds, List<DataSourceConfig> dataSourceConfigs) {
        if (isValid(ds)) {
            DataSourceConfig dataSourceConfig = getDataSourceConfig(key, ds);

            if(dataSourceConfig != null){
                dataSourceConfigs.add(dataSourceConfig);
            }
        }
    }

    private DataSourceConfig getDataSourceConfig(String key, DataSource ds){
        if(DBCP2_CLASS_EXISTS && ds instanceof BasicDataSource){
            BasicDataSource ds1 = (BasicDataSource)ds;
            return new DataSourceConfig(key, ds1.getUrl(), ds1.getUsername(), ds1.getMaxTotal(), ds1.getInitialSize(), ds1.getMaxIdle(), ds1.getMinIdle(), ds1.getMaxWaitMillis());
        } else if(HIKARI_CLASS_EXISTS && ds instanceof HikariDataSource) {
            if(HIKARI_MONITOR_EXISTS) {
                HikariDataSource ds1 = (HikariDataSource) ds;
                return new DataSourceConfig(key, ds1.getJdbcUrl(), ds1.getUsername(), ds1.getMaximumPoolSize(), -1, -1, ds1.getMinimumIdle(), ds1.getConnectionTimeout());
            } else {
                return null;
            }
        } else {
            throw new IllegalStateException("[notifyme]will not be here");
        }
    }

    @Override
    public synchronized void destroy() throws Exception {
        if (duibaDataSourceMonitorThread != null) {
            duibaDataSourceMonitorThread.interrupt();
        }
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class DataSourceConfig {

        private String dataSourceName;

        private String url;

        private String username;

        private int maxTotal;

        private int initialSize;

        private int maxIdle;

        private int minIdle;

        private long maxWaitMillis;

    }

    public static class MonitorObj {

        private short numActive;
        private short numAllocated;
        private short maxTotal;
        private short threadsAwaitingConnection;//等待连接池的线程数，-1表示不支持显示此值

        public MonitorObj(int numIdleConnections, int numActiveConnections, int maxTotal, int threadsAwaitingConnection) {
            this.numActive = (short) numActiveConnections;
            this.maxTotal = (short) maxTotal;
            this.numAllocated = (short) (numActiveConnections + numIdleConnections);
            this.threadsAwaitingConnection = (short) threadsAwaitingConnection;
        }

        public MonitorObj() {
        }

        public short getNumActive() {
            return numActive;
        }

        public short getNumAllocated() {
            return numAllocated;
        }

        public short getMaxTotal() {
            return maxTotal;
        }

        public short getThreadsAwaitingConnection() {
            return threadsAwaitingConnection;
        }
    }

}
