package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.RedisFlowReplaySpan;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;

import java.lang.reflect.Method;

/**
 * 录制 RedisTemplate.executePipelined 方法
 * 该方法底层没有调用 binaryJedis 所以需要单独录制
 * Created by guoyanfei .
 * 2019-07-12 .
 */
@Slf4j
@Aspect
public class RecordRedisTemplatePlugin {

    @Around("execution(* org.springframework.data.redis.core.RedisTemplate.*(..))")
    public Object redisTemplateJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();
        // 目前只拦截executePipelined方法
        if (!"executePipelined".equals(method.getName())) {
            return joinPoint.proceed();
        }

        if (!FlowReplayTrace.isTraced()) {
            return joinPoint.proceed();
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return joinPoint.proceed();
        }

        Object[] parameterValues = joinPoint.getArgs();
        Object obj = joinPoint.getTarget();
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(obj, method.getName(), parameterValues);

        Object ret;
        try {
            ret = joinPoint.proceed();
        } catch (Throwable t) {
            // 如果捕捉到异常，并且正在录制中，那么不录这个用例
            FlowReplayTrace.remove();
            IgnoreSubInvokesContext.unmark();
            throw t;
        }

        try {
            RedisFlowReplaySpan span = RedisFlowReplaySpan.createSpan(method, null, null, ret);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
            log.debug("RedisTemplate录制executePipelined方法_traceId={}_spanId={}", span.getTraceId(), span.getSpanId());
        } catch (Throwable t) {
            log.error("RedisTemplate录制executePipelined方法_录制异常", t);
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        return ret;
    }
}
