package cn.com.duibaboot.ext.autoconfigure.rocketmq;

import ch.qos.logback.classic.Level;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.EurekaInstanceChangedEvent;
import cn.com.duibaboot.ext.autoconfigure.core.EarlyClose;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.discovery.EurekaClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.MessageListener;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly;
import org.apache.rocketmq.client.exception.MQClientException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.EventListener;
import org.springframework.util.ReflectionUtils;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.List;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * 这个类是对DefaultMQPushConsumer的包装，主要是为了提供这个功能：在棱镜控制台中把权重调为0或禁用实例之后，当前实例停止消费消息；当启用实例且权重大于0时，当前实例重新开始消费消息。
 */
@Slf4j
public class DefaultMQPushConsumerWrapper extends EarlyClose {
    static{
        ((ch.qos.logback.classic.Logger)log).setLevel(Level.INFO);
    }

    @Resource
    private ApplicationContext applicationContext;

    private DefaultMQPushConsumer rocketMqConsumer;
    private RocketMqProperties rocketMqProperties;
    private RocketMqProperties.ConsumerProperties consumerProperties;
    private String messageListenerBeanId;

    private boolean started = false;

    private String currentAppName;
    @Value("${server.port}")
    private int serverPort;
    @Autowired(required = false)
    private EurekaClient eurekaClient;

    @Value("${spring.application.name}")
    public void setCurrentAppName(String currentAppName){
        //eureka中应用名是大写，故这里需要转成大写
        this.currentAppName = currentAppName.toUpperCase();
    }

    public DefaultMQPushConsumerWrapper(RocketMqProperties rocketMqProperties, int consumerPropertiesIndex){
        this.rocketMqProperties = rocketMqProperties;
        if(consumerPropertiesIndex == -1){//默认consumer
            consumerProperties = rocketMqProperties.getConsumer();
            messageListenerBeanId = "bootRocketMqMessageListener";
        }else{//extra-consumer
            consumerProperties = rocketMqProperties.getExtraConsumer()[consumerPropertiesIndex];
            messageListenerBeanId = "extraBootRocketMqMessageListener" + consumerPropertiesIndex;
        }
    }

    //这个方法名不能用start，以防止被EarlyClose.start过早启动
    public synchronized void startRun() throws MQClientException {
        if(started){
            log.warn("rocketmq consumer have already started, please don't call start again");
            return;
        }
        MessageListener rocketMqMessageListener;

        try {
            rocketMqMessageListener = applicationContext.getBean(messageListenerBeanId, MessageListener.class);
        }catch(NoSuchBeanDefinitionException e){
            log.error("", new IllegalStateException("bean id:[" + messageListenerBeanId + "] class:[org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently] is not exist in spring's context, 请声明，否则不会启用rocketmq消费!!!，请参考: http://gitlab2.dui88.com/basic_platform/spring-boot-ext/wikis/Starter%E7%89%B9%E6%80%A7/_rocketmq%E3%80%90%E5%BC%80%E6%BA%90RocketMQ%E3%80%91"));
            return;
        }

        /**
         * 一个应用创建一个Consumer，由应用来维护此对象，可以设置为全局对象或者单例<br>
         * 注意：consumerGroup 需要由应用来保证唯一
         * consumerGroup建议使用应用名。RocketMQ使用consumerGroup作为负载均衡的粒度，
         * 所以同一个consumerGroup订阅的Topic必须完全相同。
         * 考虑这样的场景，活动中心之前已经订阅了一些topic，某次发布新版需要增加一个topic的订阅，发布一部分服务器的情况下，新的topic的某些queue不会被消费，所以需要确保活动中心全发。某个版本如果新增了topic，则不能做灰度发布。考虑到代码可读性和性能，这个代价是值得的
         */
        //DefaultMQPullConsumer consumer = new DefaultMQPullConsumer();
        DefaultMQPushConsumer consumer = new DefaultMQPushConsumer(consumerProperties.getGroup());
        //consumer.setConsumerGroup();
        consumer.setNamesrvAddr(rocketMqProperties.getNameSrvAddr());
        consumer.setMaxReconsumeTimes(consumerProperties.getMaxReconsumeTimes());//最大重复消费次数，超过这个次数还没被确认则丢弃消息，请根据消息重要程度设置
        if(consumerProperties.getConsumeThreadNums() > 0) {
            consumer.setConsumeThreadMin(consumerProperties.getConsumeThreadNums());//消费最小线程数
            consumer.setConsumeThreadMax(Math.max(consumerProperties.getConsumeThreadNums(), 64));//消费最大线程数
        }
        //指定集群消费模型/广播消费模型
        consumer.setMessageModel(consumerProperties.getMessageModelEnum());
        //consumer.setConsumeTimeout(15);
        //consumer.setPullBatchSize(32);
        consumer.setConsumeMessageBatchMaxSize(consumerProperties.getConsumeMessageBatchMaxSize());//监听器每次接受本地队列的消息是多少条(从服务器获取还是多条的，只是本地MessageListener每次处理只有一条)
        //consumer.setMessageModel(MessageModel.BROADCASTING);
        //consumer.setPullInterval();
        //consumer.setInstanceName("Consumber");

        String topics = consumerProperties.getTopics();
        String[] topicArr = topics.split(",");

        for(String topic : topicArr) {
            consumer.subscribe(topic, "*");
        }
        //尽量使用MessageListenerConcurrently来消费，吞吐量更高，消费无顺序
        //当对消息消费顺序有严格要求时，请使用MessageListenerOrderly
        if(rocketMqMessageListener instanceof MessageListenerConcurrently) {
            consumer.registerMessageListener((MessageListenerConcurrently)rocketMqMessageListener);
        }else if(rocketMqMessageListener instanceof MessageListenerOrderly) {
            consumer.registerMessageListener((MessageListenerOrderly)rocketMqMessageListener);
        }

        rocketMqConsumer = consumer;

        rocketMqConsumer.start();

        started = true;
    }

    public synchronized void shutdown() {
        if(!started){
            return;
        }

        rocketMqConsumer.shutdown();
        if(rocketMqConsumer != null) {
            rocketMqConsumer.shutdown();
            awaitShutdown(rocketMqConsumer, 3);
        }

        started = false;
    }

    /**
     * 通过反射取出主线程池并等待一会,优雅停机
     * @param defaultMQPushConsumer
     * @param awaitMaxSeconds
     */
    private void awaitShutdown(DefaultMQPushConsumer defaultMQPushConsumer, int awaitMaxSeconds){
        Field fieldDefaultMQPushConsumerImpl = ReflectionUtils.findField(defaultMQPushConsumer.getClass(), "defaultMQPushConsumerImpl");
        fieldDefaultMQPushConsumerImpl.setAccessible(true);
        Object defaultMQPushConsumerImpl = ReflectionUtils.getField(fieldDefaultMQPushConsumerImpl, defaultMQPushConsumer);

        Field fieldConsumeMessageService = ReflectionUtils.findField(defaultMQPushConsumerImpl.getClass(), "consumeMessageService");
        fieldConsumeMessageService.setAccessible(true);
        Object consumeMessageService = ReflectionUtils.getField(fieldConsumeMessageService, defaultMQPushConsumerImpl);

        Field fieldConsumeExecutor = ReflectionUtils.findField(consumeMessageService.getClass(), "consumeExecutor");
        fieldConsumeExecutor.setAccessible(true);
        ThreadPoolExecutor consumeExecutor = (ThreadPoolExecutor)ReflectionUtils.getField(fieldConsumeExecutor, consumeMessageService);

        try {
            consumeExecutor.awaitTermination(awaitMaxSeconds, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            //Ignore
            Thread.currentThread().interrupt();
        }
    }

    synchronized boolean isStarted() {
        return started;
    }

    //监听eureka实例状态变更事件，检测当前实例状态变化，如果当前实例被禁用或权重为0，则停止消费rocketmq消息；否则消费rocketmq消息.
    @EventListener(EurekaInstanceChangedEvent.class)
    public void onEurekaInstanceChanged(EurekaInstanceChangedEvent event) throws MQClientException {
        InstanceInfo.InstanceStatus currentInstanceStatus = null;
        String weight = null;
        if(event.getAppNames().contains(currentAppName)){
            List<InstanceInfo> instances = eurekaClient.getApplication(currentAppName).getInstancesAsIsFromEureka();
            for(InstanceInfo info : instances){
                if(NetUtils.getLocalIp().equals(info.getIPAddr()) && serverPort == info.getPort()){
                    currentInstanceStatus = info.getStatus();//当前实例在eureka中的状态
                    weight = info.getMetadata().getOrDefault(DiscoveryMetadataAutoConfiguration.WEIGHT_KEY, "100");//当前实例在eureka中的权重
                }
            }
        }

        if((currentInstanceStatus == InstanceInfo.InstanceStatus.OUT_OF_SERVICE || "0".equals(weight))
                && this.isStarted()){
            log.info("检测到当前实例被禁用(或权重调为0), 停止本实例的rocketmq消费");
            this.shutdown();
        } else if((currentInstanceStatus == InstanceInfo.InstanceStatus.UP && !"0".equals(weight))
                && !this.isStarted()){
            log.info("检测到当前实例被重新启用且权重大于0, 重新开始本实例的rocketmq消费");
            this.startRun();
        }
    }

    //EarlyClose
    @Override
    public void stop() {
        this.shutdown();
    }

    //EarlyClose
    //和onsConsumer一起关闭，且在统一线程池之前关闭
    @Override
    public int getPhase() {
        return 1;
    }

}
