package cn.com.duibaboot.ext.autoconfigure.devtools;

import cn.com.duiba.boot.utils.RequestUtils;
import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import com.alibaba.fastjson.JSONObject;
import org.apache.commons.lang.ArrayUtils;
import org.springframework.boot.devtools.restart.Restarter;
import org.springframework.context.ApplicationContext;
import org.springframework.core.env.Environment;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.net.URL;
import java.util.Set;

/**
 * DevTools拦截器, 配合BootReload chrome插件使用，提高开发效率
 */
public class BootDevToolsFilter implements Filter{

    /**
     * 是否支持热重启
     */
    private boolean canHotRestart = false;
    /**
     * 不支持热重启时，这个字段表示不支持的原因
     */
    private String errorMsg;

    /**
     * 是否正在热重启
     */
    private volatile boolean isHotRestarting = false;

    private ClassPathChangedFilesContainer classPathChangedFilesContainer;

    public BootDevToolsFilter(ApplicationContext applicationContext, ClassPathChangedFilesContainer classPathChangedFilesContainer){
        this.classPathChangedFilesContainer = classPathChangedFilesContainer;
        Environment environment = applicationContext.getEnvironment();

        if(SpringBootUtils.isJarInJarMode()){
            errorMsg = "检测到你的应用在java -jar模式下运行，该模式无法使用热重启功能，如需使用热重启，请在IDEA、Eclipse等IDE中直接运行";
        } else if("false".equals(environment.getProperty("spring.devtools.restart.enabled"))){
            errorMsg = "检测到你的应用配置了spring.devtools.restart.enabled=false，这表示禁用了开发时热重启功能；如需使用热重启，请移除该配置。（你可以放心移除该配置，这个配置只影响开发环境）";
        } else if(isDevToolsNotInClassPath() || classPathChangedFilesContainer == null){
            errorMsg = "没有检测到spring-boot-devtools依赖(热重启依赖于之)，请检查您是否升级到了最新的兑吧spring-boot-ext依赖，或者您是否排除了该依赖。";
        } else{
            canHotRestart = true;
        }
    }

    @Override
    public void init(FilterConfig filterConfig) {
        //do nothing
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest)servletRequest;
        HttpServletResponse response = (HttpServletResponse)servletResponse;
        String uri = RequestUtils.getRequestPath(request);
        if(uri.equals("/devtools/check")){//检查devtools是否启用
            if(!canHotRestart){
                error(errorMsg, response);
                return;
            }
            if(isHotRestarting){//TODO 这个目前没法支持判断是否在热重启，因为热重启时端口是关闭的，后续考虑单独开个随机端口的内置jetty服务器来做这个判断。
                error("正在热重启中,请稍候...", response);
                return;
            }

            error("检测到当前应用支持热重启, 如需要热重启，请点击下面的按钮", response);
            return;
        }else if(uri.equals("/devtools/restart")){//执行热重启
            if(!canHotRestart){
                error(errorMsg, response);
                return;
            }
            if(isHotRestarting){//TODO 这个目前没法支持判断是否在热重启，因为热重启时端口是关闭的，后续考虑单独开个随机端口的内置jetty服务器来做这个判断。
                error("正在热重启中,请稍候...", response);
                return;
            }

            Set<?> changedFilesSet = classPathChangedFilesContainer.getChangedFilesAndClear();

            //某些情况下由于IDEA的bug或其他原因，获取不到需要扫描的url，这种情况下强制重启
            URL[] scanUrls = Restarter.getInstance().getInitialUrls();

            if(changedFilesSet.isEmpty() && !ArrayUtils.isEmpty(scanUrls)){
                error("<span style='font-size:20px;'>没有类发生变动，不需要热重启</span><br/>(如果你修改了java代码，在IDEA中需要先手动点击菜单【Build -> Build Project】触发编译代码;<br/>在Eclipse中则只需保存代码即可自动触发编译代码;<br/>代码编译完成后再到这里重启)", response);
                return;
            }

            isHotRestarting = true;
            error("成功请求热重启，开始执行，请关注IDE控制台。", response);

            //发出请求重启事件
            classPathChangedFilesContainer.requestRestart(changedFilesSet);
        }else{
            filterChain.doFilter(servletRequest, servletResponse);
        }
    }

    public void error(String message, HttpServletResponse response) throws IOException {
        JSONObject json = new JSONObject();
        json.fluentPut("success", "true")
                .fluentPut("canHotRestart", canHotRestart)
                .fluentPut("hotRestarting", isHotRestarting)
                .fluentPut("message", message);

        response.setContentType("application/json;charset=UTF8");
        response.getWriter().write(json.toString());
        response.getWriter().flush();
        response.getWriter().close();
    }

    @Override
    public void destroy() {
        //do nothing
    }

    private boolean isDevToolsNotInClassPath(){
        try {
            Class.forName("org.springframework.boot.devtools.restart.classloader.RestartClassLoader");
            return false;
        } catch (ClassNotFoundException e) {
            return true;
        }
    }

}
