package cn.com.duibaboot.ext.autoconfigure.etcd;

import cn.com.duibaboot.ext.autoconfigure.etcd.client.EtcdKVClientDelegate;
import cn.com.duibaboot.ext.autoconfigure.etcd.client.JetcdKVClientDelegate;
import cn.com.duibaboot.ext.autoconfigure.etcd.client.RestEtcdKVClientDelegate;
import cn.com.duibaboot.ext.autoconfigure.etcd.properties.EtcdProperties;
import com.coreos.jetcd.Client;
import org.springframework.boot.actuate.autoconfigure.endpoint.condition.ConditionalOnEnabledEndpoint;
import org.springframework.boot.actuate.endpoint.annotation.Endpoint;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

@Configuration
@ConditionalOnProperty(value = "duiba.cloud.etcd.enabled", matchIfMissing = true)
@EnableConfigurationProperties(EtcdProperties.class)
public class EtcdAutoConfiguration {

    @Configuration
    @ConditionalOnClass(Client.class)
    @ConditionalOnMissingBean(Client.class)
    public static class JetcdConfiguration{
        @Bean
        public Client etcdClient(EtcdProperties etcdProperties) {
            return Client.builder().endpoints(etcdProperties.getUris()).build();
        }

        @Bean
        @ConditionalOnMissingBean(EtcdKVClientDelegate.class)
        public EtcdKVClientDelegate etcdKVClientDelegate(Client etcdClient) {
            return new JetcdKVClientDelegate(etcdClient);
        }

        @Bean
        @ConditionalOnMissingBean
        @ConditionalOnEnabledEndpoint
        public EtcdEndpoint etcdEndpoint(Client etcdClient, EtcdProperties etcdProperties) {
            return new EtcdEndpoint(etcdClient, etcdProperties);
        }
    }

    //由于jetcd依赖到了3.5.1版本的protobuf，而hbase依赖2.5.0版本的protobuf，有jar包冲突的问题（用shade插件也不好解决，因为protobuf包里有很多资源文件）
    //所以部分依赖了hbase的项目，我们允许他们exclude掉jetcd的jar包，这样的话我们就需要自己封装一个简单版本的rest etcd客户端。
    @Configuration
    @ConditionalOnMissingClass("com.coreos.jetcd.Client")
    public static class SimpleRestEtcdClientConfiguration{

        @Bean
        @ConditionalOnMissingBean(EtcdKVClientDelegate.class)
        public EtcdKVClientDelegate etcdKVClientDelegate(EtcdProperties etcdProperties) {
            String[] etcdUris = etcdProperties.getUris().toArray(new String[etcdProperties.getUris().size()]);
            return new RestEtcdKVClientDelegate(etcdUris);
        }
    }

    @Configuration
    @ConditionalOnClass(Endpoint.class)
    protected static class EtcdHealthConfig {

        @Bean
        @ConditionalOnMissingBean
        @ConditionalOnEnabledEndpoint
        public EtcdLastRefreshStateEndpoint etcdLastRefreshStateEndpoint() {
            return new EtcdLastRefreshStateEndpoint();
        }

//        @Bean
//        @ConditionalOnMissingBean
//        @ConditionalOnEnabledHealthIndicator("etcd")
//        public EtcdHealthIndicator etcdHealthIndicator(Client etcdClient) {
//            return new EtcdHealthIndicator(etcdClient);
//        }
    }
}
