package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import cn.com.duiba.wolf.utils.UUIDUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.RecordContextHolder;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.IgnoreSubInvokesContext;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.io.Serializable;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * 流量录制用于回归的trace，一个trace表示一次调用的整条链路
 * Created by guoyanfei .
 * 2019-01-24 .
 */
@Slf4j
@Data
public class FlowReplayTrace implements Serializable {

    private static final long serialVersionUID = -622886226087802579L;

    /**
     * trace的唯一标识
     */
    private String traceId;

    private FlowReplaySpan mainSpan;

    private List<FlowReplaySpan> subSpans;

    /**
     * 录制开始后，距离录制开始时间的秒数
     */
    private Short secondsAfterStart;

    private static class ThreadLocalHolder {

        private ThreadLocalHolder() {
        }

        private static final ThreadLocal<FlowReplayTrace> threadLocal2FlowReplayTrace = new ThreadLocal<>();
    }

    /**
     * 通过主调用创建trace
     * @param mainSpan
     */
    public static synchronized void createTrace(FlowReplaySpan mainSpan) {
        if (mainSpan == null) {
            return;
        }
        Long recordStartTime = RecordContextHolder.getStartTime();
        if (recordStartTime == null) {
            return;
        }
        FlowReplayTrace c = new FlowReplayTrace();
        c.traceId = UUIDUtils.createUUID();
        c.mainSpan = mainSpan;
        c.subSpans = Collections.synchronizedList(new LinkedList<>());
        c.secondsAfterStart = (short) ((System.currentTimeMillis() - recordStartTime) / 1000);
        ThreadLocalHolder.threadLocal2FlowReplayTrace.set(c);
    }

    /**
     * 新增子调用
     * @param subSpan
     */
    public static synchronized void addSubSpan(FlowReplaySpan subSpan) {
        FlowReplayTrace trace = ThreadLocalHolder.threadLocal2FlowReplayTrace.get();
        if (trace == null || trace.subSpans == null) {
            return;
        }
        trace.subSpans.add(subSpan);
    }

    /**
     * 获取当前的traceId
     * @return
     */
    public static String getCurrentTraceId() {
        FlowReplayTrace trace = ThreadLocalHolder.threadLocal2FlowReplayTrace.get();
        if (trace == null) {
            return null;
        }
        return trace.traceId;
    }

    /**
     * 当前请求是否被追踪
     * @return
     */
    public static boolean isTraced() {
        return RecordContextHolder.isRecording() && get() != null;
    }

    /**
     * 获取当前trace
     * @return
     */
    public static FlowReplayTrace get() {
        FlowReplayTrace trace = ThreadLocalHolder.threadLocal2FlowReplayTrace.get();
        if (trace == null) {
            return null;
        }
        return trace;
    }

    /**
     * 一般在异常的时候，才会直接调用这个
     */
    public static void remove() {
        ThreadLocalHolder.threadLocal2FlowReplayTrace.remove();
    }

    /**
     * 每个用例开始录制之前，都需要清理一下录制环境
     * 保持新录制的用例是干净的
     * 目前需要清理线程变量
     */
    public static void cleanRecordEnv() {
        FlowReplayTrace.remove();
        IgnoreSubInvokesContext.unmark();
    }
}
