package cn.com.duibaboot.ext.autoconfigure.httpclient;

import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInvocation;
import org.springframework.context.ApplicationContext;
import org.springframework.context.event.ContextClosedEvent;
import org.springframework.context.event.EventListener;

import javax.annotation.Resource;

/**
 * 一个HttpClient方法拦截器的实现，用于禁止使用者调用统一HttpClient.close方法
 */
@Slf4j
class HttpClientCloseMethodInterceptor implements HttpClientMethodInterceptor {

	/**
	 * 标记是否允许关闭统一HttpClient，只有spring容器关闭时才允许关闭HttpClient
	 */
	private volatile boolean canClose = false;

	@Resource
	private ApplicationContext applicationContext;

	@Override
	public Object invoke(MethodInvocation invocation) throws Throwable {
		if(!"close".equals(invocation.getMethod().getName())){
			return invocation.proceed();
		}

		if(canClose){
			return invocation.proceed();
		}

		log.warn("", new IllegalStateException("请勿在应用中调用统一HttpClient.close()方法!"));

		return null;
	}

	@Override
	public int getOrder() {
		return 1;
	}

	@EventListener(ContextClosedEvent.class)
	public void onContextClosed(ContextClosedEvent e){
		if(e.getApplicationContext() == applicationContext) {
			canClose = true;
		}
		//收到ContextClosedEvent事件后才会开始销毁bean阶段
	}

}
