package cn.com.duibaboot.ext.autoconfigure.monitor.elasticjob;

import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.elasticjob.lite.api.ElasticJob;
import io.elasticjob.lite.api.ShardingContext;
import io.elasticjob.lite.api.dataflow.DataflowJob;
import io.elasticjob.lite.api.script.ScriptJob;
import io.elasticjob.lite.api.simple.SimpleJob;
import io.elasticjob.lite.executor.ShardingContexts;
import io.elasticjob.lite.util.concurrent.ExecutorServiceObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-08-22 20:27
 * @descript:
 * @version: 1.0
 */
@Slf4j
public class ElasticjobMonitorManager {

    @Autowired
    private ApplicationContext applicationContext;

    private Map<String, String> jobCronMap = Maps.newConcurrentMap();

    private final Set<ElasticJob> runingJob = Sets.newConcurrentHashSet();

    private volatile String jobNamespace;

    private final ExecutorService executorService = new ExecutorServiceObject("monitor-job-trigger", Runtime.getRuntime().availableProcessors() * 2).createExecutorService();

    void setJobNamespace(String namespace) {
        jobNamespace = namespace;
    }

    String getJobNamespace() {
        return jobNamespace;
    }

    void putJobCronMap(String jobName, String cron) {
        jobCronMap.put(jobName, cron);
    }

    String getJobCron(String jobName) {
        return jobCronMap.get(jobName);
    }

    String trigger(String jobClassName, List<Integer> shardingItems) {

        if (CollectionUtils.isEmpty(shardingItems)) {
            return "没有分片信息";
        }

        ElasticJob elasticJob = null;
        String message = "触发作业异常";
        try {
            Class jobClass = Class.forName(jobClassName);
            Map<String, ElasticJob> beanMap = applicationContext.getBeansOfType(jobClass);
            if (beanMap == null || beanMap.size() == 0) {
                return "没有找到可以触发的job示例";
            }

            elasticJob = beanMap.entrySet().iterator().next().getValue();
            if (runingJob.contains(elasticJob)) {
                return "当前job正在运行，请稍后再试";
            }

            if(elasticJob instanceof ScriptJob) {
                return "暂时不支持手动调度ScriptJob";
            }

            runingJob.add(elasticJob);
            if (elasticJob instanceof SimpleJob) {
                simpleProcess(elasticJob, shardingItems);
            } else if (elasticJob instanceof DataflowJob) {
                dataProcess(elasticJob, shardingItems);
            }

            message = "触发作业成功";

        } catch (Exception e) {
            log.error("触发作业异常", e);
        } finally {

            if (elasticJob != null) {
                runingJob.remove(elasticJob);
            }
        }

        return message;
    }

    private void simpleProcess(ElasticJob elasticJob, List<Integer> shardingItems) {

        ShardingContexts empty = new ShardingContexts(null, null, 0, null, Maps.newHashMap());
        final CountDownLatch latch = new CountDownLatch(shardingItems.size());
        shardingItems.forEach(shardingItem -> executorService.submit(() -> {

            try {
                ((SimpleJob) elasticJob).execute(new ShardingContext(empty, shardingItem));
            } finally {
                latch.countDown();
            }
        }));

        try {
            latch.await();
        } catch (final InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }

    private void dataProcess(ElasticJob elasticJob, List<Integer> shardingItems) {

        final CountDownLatch latch = new CountDownLatch(shardingItems.size());
        final ShardingContexts empty = new ShardingContexts(null, null, 0, null, Maps.newHashMap());
        shardingItems.forEach(shardingItem -> executorService.submit(() -> {

            try {
                DataflowJob job = (DataflowJob) elasticJob;
                ShardingContext context = new ShardingContext(empty, shardingItem);
                List<?> data = job.fetchData(context);
                job.processData(context, data);
            } finally {
                latch.countDown();
            }
        }));

        try {
            latch.await();
        } catch (final InterruptedException ex) {
            Thread.currentThread().interrupt();
        }

    }
}
