package cn.com.duibaboot.ext.autoconfigure.accesslog;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.wolf.utils.SecurityUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Charsets;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.core.io.ClassPathResource;
import org.springframework.util.AntPathMatcher;

import java.text.SimpleDateFormat;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Created by xuhengfei on 16/12/27.
 */
public abstract class AbstractAccessLogFilter<R, S> implements InitializingBean {

    private static final Logger accessLog = LoggerFactory.getLogger("duiba_access_log");

    private static final Logger log=LoggerFactory.getLogger(AbstractAccessLogFilter.class);

    public static final String OW_HOST="host";
    private static final String OW_PATH="accesslog_overwrite_path";

    private static final String ModeBlack="black";//黑名单模式
    private static final String ModeWhite="white";//白名单模式

    private static final String COLLECT_COOKIE_MAP_KEY="_accesslog_collect_cookie_map";

    private static final String LOG_COOKIE_ATTRIBUTE_KEY = "_accesslog_log_cookie";
    private static final String EX_ATTRIBUTE_KEY = "_accesslog_ex";
    private static final String OVERWRITE_ATTRIBUTE_KEY = "_accesslog_overwrite";
    private static final String APP_ID_ATTRIBUTE_KEY = "_accesslog_app_id";
    private static final String CONSUMER_ID_ATTRIBUTE_KEY = "_accesslog_consumer_id";
    private static final String DONT_WRITE_CURRENT_ACCESSLOG="dont_write_current_accesslog";

    private String mode=ModeBlack;
    private Set<String> inhosts=new HashSet<>();

    private Set<String> blackPaths=new HashSet<>();
    private Set<String> blackUriPatterns = new HashSet<>();
    private Map<String,String> cookieHosts=new ConcurrentHashMap<>();
    private AntPathMatcher pathMatcher = new AntPathMatcher();

    @Override
    public void afterPropertiesSet(){
        try {
            Properties p = new Properties();
            p.load(new ClassPathResource("/accesslog_filter.properties").getInputStream());
            if(!p.isEmpty()){
                loadConfig(p);
                log.info("load accesslog_filter.properties success");
            }
        }catch (Exception e){
            log.warn("load accesslog_filter.properties failed", e);
        }
    }

    /**
     * 加载配置项
     * @param properties
     */
    private void loadConfig(Properties properties){
        mode=properties.getProperty("accesslog.mode");
        String hoststring=properties.getProperty("accesslog.hosts");
        if(hoststring!=null){
            List<String> hosts=Arrays.asList(hoststring.trim().split(","));
            if(!CollectionUtils.isEmpty(hosts)){
                inhosts=new HashSet<>(hosts);
            }
        }
        String blackpaths=properties.getProperty("accesslog.blackpaths");
        List<String> paths=Arrays.asList(StringUtils.split(StringUtils.trimToEmpty(blackpaths),","));
        for(String path : paths){
            if(path != null && !path.contains("*")){
                blackPaths.add(path);
            }else if(path != null && path.contains("*")){
                blackUriPatterns.add(path);
            }
        }
    }

    /**
     * 是否需要进行日志打印
     * @return
     */
    private boolean needLog(R req){
        Boolean val = (Boolean)getAttribute(req, DONT_WRITE_CURRENT_ACCESSLOG);
        if(val != null && val){
            return false;
        }
        String host=getHost(req);
        if(ModeBlack.equals(mode)){
            if(inhosts.contains(host)){
                return false;
            }
        }else if(ModeWhite.equals(mode)){
            if(!inhosts.contains(host)){
                return false;
            }
        }

        String path=clearRequestURI(getPath(req));
        if(blackPaths.contains(path)){
            return false;
        }
        for(String pattern : blackUriPatterns){
            if(this.pathMatcher.match(pattern, path)){
                return false;
            }
        }

        return true;
    }

    private String getHost(R req){
        Map<String, String> map = (Map<String, String>)getAttribute(req, OVERWRITE_ATTRIBUTE_KEY);
        if(map!=null){
            String host=map.get(OW_HOST);
            if(host!=null){
                return host;
            }
        }

        return getHeader(req, "host");
    }

    protected abstract String getRequestURI(R req);
    protected abstract String getMethod(R req);
    protected abstract String getQueryString(R req);
    protected abstract String getHeader(R req, String key);
    protected abstract String getIpAddr(R req);
    protected abstract boolean isPerfTestRequest(R req);
    protected abstract void addAttribute(R req, String key, Object value);
    protected abstract Object getAttribute(R req, String key);
    protected abstract String getParameter(R req, String key);
    protected abstract List<String> getParameterNames(R req);
    protected abstract int getStatus(S resp);
    protected abstract void addCookie(S resp, String key, String value, String domain, String path);

    protected void doBefore(R req, S resp) {
        Map<String, String> collCookie=new HashMap<>();
        try {
            //在执行filter之前种cookie,避免resp被提前关闭,无法写入cookie
            addCookieIfNeed(req, resp, collCookie);
            addAttribute(req, COLLECT_COOKIE_MAP_KEY, collCookie);
        }catch(Exception e){
            log.error("addCookieIfNeed error");
        }
    }

    protected void doAfter(R req, S resp, long cost) {
        try {
            //判断是否性能压测url
            boolean isPerfUrl = isPerfTestRequest(req);

            if(!isPerfUrl && needLog(req)){
                Map<String, String> collCookie = (Map<String, String>)getAttribute(req, COLLECT_COOKIE_MAP_KEY);
                processAccessLog(req,resp,cost,collCookie);
            }
        }catch (Exception e){
            log.error("AccessLogFilter process error, message=", e);
        }
    }

    private void addCookieIfNeed(R req, S resp, Map<String,String> collCookie){
        List<String> parameterNames=getParameterNames(req);
        String host=getHeader(req, "host");
        String cookieHost=getCookieHost(host);

        for(String key : parameterNames){
            if(!key.startsWith("tck_")){
                continue;
            }

            String[] v=key.split("_");
            if(v.length!=3){
                continue;
            }

            String verify = SecurityUtils.encode2StringByMd5(v[1]);
            if(verify.endsWith(v[2])){
                String name=v[1];
                String value=getParameter(req, key);
                //验证通过
                String cookiekey="_coll_"+name;
                collCookie.put(cookiekey,value);
                //写入cookie

                try {
                    addCookie(resp, cookiekey, value, cookieHost, "/");
                }catch (Exception e){
                    log.error("addCookie error,cookieKey="+cookiekey+",value="+value,e);
                }
            }
        }
    }

    private void processAccessLog(R req, S resp,long cost,Map<String,String> collCookie) throws Exception{//NOSONAR
        //只记录get post请求
        String method = StringUtils.lowerCase(getMethod(req));
        if(!"get".equals(method) && !"post".equals(method)){
            return;
        }

        Map<String, Object> map = new HashMap<>();
        map.put("url_host", getHost(req));
        map.put("url_path", clearRequestURI(getPath(req)));
        map.put("url_query", getQueryString(req));

        map.put("http_method",getMethod(req));
        map.put("rc",getStatus(resp));
        map.put("rt",cost);
        map.put("mip", NetUtils.getLocalIp());//记录服务器IP
        putIfNotNull(map, "ex", getExPair(req));

        //如果url中有callback参数,认为此url是一个JSONP请求,标记为POST类型,不作为PV统计
        String callback=getParameter(req, "callback");
        if(!StringUtils.isBlank(callback)){
            map.put("http_method","POST");
        }

        Map<String, String> cookieMap = getCookieMap(req);

        //优先从线程变量中取数据
        Long consumerId = getConsumerId(req);

        Long appId = getAppId(req);

        String ac = cookieMap.get("_ac");
        if (ac != null && (consumerId==null || appId==null)){
            //如果还没有取到,尝试从cookie取数据
            try {
                String json = new String(SecurityUtils.decodeBase64(ac), Charsets.UTF_8);
                if(json!=null){
                    JSONObject object = JSONObject.parseObject(json);
                    if(appId==null){
                        appId = object.getLong("aid");
                    }
                    if(consumerId==null){
                        consumerId = object.getLong("cid");
                    }
                }
            }catch(Exception e){
                log.info("_ac decode fail , _ac= "+ac);
            }
        }

        putIfNotNull(map, "consumer_id", consumerId);

        putIfNotNull(map, "app_id", appId);

        handleCookies(cookieMap, collCookie);

        handleParams(req, collCookie);

        //支持线程变量中取cookie进行日志打印,便于不同系统的一些定制需求,此为cookie的最高优先级获取方法
        handleLogCookie(req, collCookie);

        putIfNotEmpty(map, "cookie", collCookie);

        putUa(req, map);

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        map.put("time", sdf.format(new Date()));
        String referer = getHeader(req, "referer");
        putIfNotNull(map, "referer", referer);
        String ip = getIpAddr(req);
        putIfNotNull(map, "ip", ip);

        accessLog.info(JSON.toJSONString(map));
    }

    private Map<String, Object> getExPair(R req){
        return (Map<String, Object>)getAttribute(req, EX_ATTRIBUTE_KEY);
    }

    private void putUa(R req, Map<String, Object> map){
        String ua = StringUtils.trimToEmpty(getHeader(req, "user-agent"));
        if (ua.length() > 500) {
            ua = ua.substring(0, 499);
        }
        if (!StringUtils.isEmpty(ua)) {
            map.put("user_agent", ua);
        }
    }

    private void putIfNotNull(Map<String, Object> map, String key, Object value){
        if(value != null){
            map.put(key, value);
        }
    }

    private void putIfNotEmpty(Map<String, Object> map, String key, Map<String,String> collCookie){
        if(!collCookie.isEmpty()){
            map.put(key, collCookie);
        }
    }

    protected abstract Map<String, String> getCookieMap(R req);

    private void handleCookies(Map<String, String> cookieMap,Map<String,String> collCookie){
        for(Map.Entry<String, String> entry : cookieMap.entrySet()){
            String key = entry.getKey();
            String cookie = entry.getValue();
            if(key.startsWith("_coll_")){
                collCookie.put(key,cookie);
            }
        }
    }

    private void handleParams(R req, Map<String,String> collCookie){
        //特殊处理
        String slotId=getParameter(req, "adslotId");
        if(slotId!=null){
            collCookie.put("_coll_slot",slotId);
        }
        String deviceId=getParameter(req, "deviceId");
        if(deviceId!=null){
            collCookie.put("_coll_device",deviceId);
        }
    }

    private void handleLogCookie(R req, Map<String,String> collCookie){
        Map<String,String> logCookie=getLogCookie(req);
        if(logCookie!=null){
            for(Map.Entry<String, String> entry : logCookie.entrySet()){
                collCookie.put(entry.getKey(),entry.getValue());
            }
        }
    }

    private Map<String, String> getLogCookie(R req) {
        return (Map<String, String>) getAttribute(req, LOG_COOKIE_ATTRIBUTE_KEY);
    }

    /**
     * requestURI中可能出现连续两个/，比如//activity//index，需要处理下转为一个/
     * @param requestURI
     * @return
     */
    private String clearRequestURI(String requestURI){
        if(StringUtils.isBlank(requestURI)){
            return requestURI;
        }

        return StringUtils.replace(requestURI, "//", "/");
    }

    /**
     * 根据host查询 通配符cookie地址
     * @param host
     * @return
     */
    private String getCookieHost(String host){
        if(host==null){
            return null;
        }
        if(host.contains(":")){
            host=host.substring(0,host.indexOf(':'));
        }
        String cookieHost=cookieHosts.get(host);
        if(cookieHost!=null){
            return cookieHost;
        }
        String hostLow=host.toLowerCase();
        if(hostLow.endsWith(".duiba.com.cn") ){
            cookieHost="duiba.com.cn";
        }else if(hostLow.endsWith(".dui88.com")){
            cookieHost="dui88.com";
        }else if(hostLow.endsWith(".duibar.com")){
            cookieHost="duibar.com";
        }else if(hostLow.endsWith(".tuia.cn")) {
            cookieHost = "tuia.cn";
        }

        if(cookieHost!=null){
            cookieHosts.put(host,cookieHost);
        }
        return cookieHost;
    }

    /**
     * 设置appId
     * @param req
     * @param appId
     */
    public void setAppId(R req, Long appId) {
        ensureSafeCall();
        addAttribute(req, APP_ID_ATTRIBUTE_KEY, appId);
    }

    /**
     * 设置consumerId
     * @param req
     * @param consumerId
     */
    public void setConsumerId(R req, Long consumerId) {
        ensureSafeCall();
        addAttribute(req, CONSUMER_ID_ATTRIBUTE_KEY, consumerId);
    }

    private Long getAppId(R req) {
        Long appId = (Long)getAttribute(req, APP_ID_ATTRIBUTE_KEY);
        return appId;
    }

    private Long getConsumerId(R req){
        Long consumerId = (Long)getAttribute(req, CONSUMER_ID_ATTRIBUTE_KEY);
        return consumerId;
    }

    /**
     * 添加cookie信息到线程变量,日志优先从线程变量中获取
     * @param key
     * @param value
     */
    public void putLogCookie(R req, String key,String value){
        ensureSafeCall();
        Map<String, String> map = (Map<String, String>)getAttribute(req, LOG_COOKIE_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            addAttribute(req, LOG_COOKIE_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    /**
     * 添加额外信息到线程变量,日志从线程变量获取信息进行打压输出
     * @param key
     * @param value
     */
    public void putExPair(R req, String key,Object value){
        ensureSafeCall();
        Map<String, Object> map = (Map<String, Object>)getAttribute(req, EX_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            addAttribute(req, EX_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    /**
     * 添加host等覆盖信息到线程变量, 打印日志时优先从此处获取
     * @param key
     * @param value
     */
    public void putOverWritePair(R req, String key,String value){
        ensureSafeCall();
        Map<String, String> map = (Map<String, String>)getAttribute(req, OVERWRITE_ATTRIBUTE_KEY);
        if(map == null){
            map = new HashMap<>();
            addAttribute(req, OVERWRITE_ATTRIBUTE_KEY, map);
        }

        map.put(key, value);
    }

    /**
     * 当前请求不写accesslog，该设置只对本次请求有效
     */
    public void dontWriteCurrentLog(R req){
        addAttribute(req, DONT_WRITE_CURRENT_ACCESSLOG, true);
    }

    private String getPath(R req){
        String path = (String)getAttribute(req, OW_PATH);
        if(StringUtils.isBlank(path)){
            path = getRequestURI(req);
        }

        return path;
    }

    /**
     * 设置path，设置后在记录path时会优先使用这里设置的值
     * @param path
     */
    public void setOverWritePath(R req, String path){
        this.addAttribute(req, OW_PATH, path);
    }

    /**
     * 确保AccessLogFilter的方法只能被servlet/请求所在线程调用，否则抛出异常，以避免误用
     */
    protected abstract void ensureSafeCall();
}
