package cn.com.duibaboot.ext.autoconfigure.httpclient;

import brave.httpclient.TracingHttpClientBuilder;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.httpclient.ssre.SsreBeanPostProcessor;
import com.google.common.collect.Sets;
import org.apache.http.*;
import org.apache.http.client.HttpClient;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestExecutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.web.client.RestTemplateCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.io.IOException;

/**
 * 自动构建HttpClient
 */
@Configuration
@ConditionalOnClass(HttpClient.class)
@AutoConfigureBefore(name="org.springframework.cloud.netflix.feign.ribbon.FeignRibbonClientAutoConfiguration")
public class HttpClientAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(HttpClientAutoConfiguration.class);

    @Configuration
    @ConditionalOnClass(TracingHttpClientBuilder.class)
    public static class DuibaTracingHttpClientBuilderConfiguration{

        @Bean
        public HttpClientBuilder httpClientBuilder(){
            return TracingHttpClientBuilder.create();
        }
    }

    @Configuration
    @ConditionalOnMissingClass("brave.httpclient.TracingHttpClientBuilder")
    public static class DuibaHttpClientBuilderConfiguration{

        @Bean
        public HttpClientBuilder httpClientBuilder(){
            return HttpClientBuilder.create();
        }
    }

    //默认情况下不能调用CloseableHttpResponse.close()否则tcp长连接会关闭,但是这里构造的httpClient可以这样调用，因为经过aop处理过。
    //这个httpClient可以被业务使用，但是由于这个httpClient也会被FeignClient使用，所以注意有些参数不能设置过小
    @Bean(name="apacheHttpClient", destroyMethod = "close") //这里名字不能用httpClient，以防和spring-cloud-gateway中默认创建的异步HttpClient名字冲突。
    public CloseableHttpClient apacheHttpClient(HttpClientBuilder httpClientBuilder){
        return httpClientBuilder
                .setDefaultRequestConfig(RequestConfig.custom().setConnectTimeout(5000).setSocketTimeout(60000).setConnectionRequestTimeout(10).build())
                .setMaxConnPerRoute(100)
                .setMaxConnTotal(5000)//一定要设置maxConnTotal，不然默认是10
                .setUserAgent("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_5) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/59.0.3071.115 Safari/537.36")//userAgent仿真，以防第三方设置的WAF判断了UA从而调用失败.
                .disableAutomaticRetries()//禁止重试
                .disableCookieManagement()
                .useSystemProperties()//for proxy
                .setKeepAliveStrategy(new DefaultConnectionKeepAliveStrategy(){
                    @Override
                    public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                        long time = super.getKeepAliveDuration(response, context);
                        if(time == -1){
                            time = 30000;//链接最多空闲30秒
                        }
                        return time;
                    }
                }).setRequestExecutor(new HttpRequestExecutor(){
                    @Override
                    public HttpResponse execute(HttpRequest request, HttpClientConnection conn, HttpContext context) throws IOException, HttpException {
                        if(RpcContext.hasContext() && conn instanceof HttpInetConnection) {
                            HttpInetConnection c = (HttpInetConnection)conn;
                            if(RpcContext.hasContext()) {
                                RpcContext rc = RpcContext.getContext();
                                rc.setLocalAddr(c.getLocalAddress().getHostAddress() + ":" + c.getLocalPort());
                                rc.setRemoteAddr(c.getRemoteAddress().getHostAddress() + ":" + c.getRemotePort());
                                if (logger.isDebugEnabled() && rc.getMethod()!=null) {
                                    logger.debug("invoking {}.{}, {} -> {}",
                                            rc.getMethod().getDeclaringClass().getSimpleName(),
                                            rc.getMethod().getName(),
                                            rc.getLocalAddr(),
                                            rc.getRemoteAddr());
                                }
                            }
                        }
                        return super.execute(request, conn, context);
                    }
                })
                .evictExpiredConnections()//每隔10秒主动扫描并逐出超时的连接（超过keepAliveTimeout）
//                .setConnectionTimeToLive(30, TimeUnit.SECONDS)//这个要么不设，要么就设大点，这个用于设置连接最长保留多长时间（从创建开始计算）。而keepalive策略会在连接被使用后重新计算存活时间
                .build();
    }

    @Bean
    public HttpClientHystrixMethodInterceptor httpClientHystrixMethodInterceptor(){
        return new HttpClientHystrixMethodInterceptor();
    }
    @Bean
    public HttpClientCloseMethodInterceptor httpClientCloseMethodInterceptor(){
        return new HttpClientCloseMethodInterceptor();
    }

    /**
     * 给httpClient加上熔断功能
     */
    @Bean
    public static SpecifiedBeanPostProcessor httpClientAopPostProcessor(){
        return new HttpClientAopPostProcessor();
    }

    @Bean
    public RestTemplateCustomizer duibaRestTemplateCustomizer(){
        return new DuibaRestTemplateCustomizer();
    }

    /**
     * ssre防御，阻止httpclient访问内网
     * @return
     */
    @Bean
    @ConditionalOnProperty(name="duiba.security.ssre.enable", havingValue = "true", matchIfMissing = true)
    @ConditionalOnMissingBean
    public static SsreBeanPostProcessor ssreBeanFactoryPostProcessor(){
        return new SsreBeanPostProcessor();
    }

}
