package cn.com.duibaboot.ext.autoconfigure.perftest;

import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.env.Environment;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.util.Assert;

import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

/**
 * 自动根据当前是不是压测请求决定访问原始数据库或者影子库
 * Created by wenqi.huang on 2016/11/24.
 */
@Slf4j
public abstract class PerfTestRoutingDataSource<DS extends DataSource> extends AbstractRoutingDataSource implements DisposableBean {

    /**
     * 原始库的key
     */
    private static final String ORIGINAL_DATASOURCE = "original";
    /**
     * 影子库的key
     */
    private static final String TEST_DATASOURCE = "test";
    /**
     * 影子库的scheme前缀
     */
    protected static final String TEST_SCHEME_PREFIX = "perf__";
    private static final String SHADE_SCHEMA = ".shade";

    private volatile boolean hasTestDataSource = false;
    private volatile boolean testDataSourceInitted = false;
    /**
     * 原始库的sheme
     */
    private String originalScheme;

    /**
     * 原始库datasource
     */
    private DS originalDataSource;
    private Environment environment;
    private volatile DS shadeDataSource;
    private ConcurrentMap<Object, Object> targetDataSources = new ConcurrentHashMap<>();
    private boolean thisInitted = false;
    private ConcurrentMap<String, String> normalDbUrl2shadeUrlMap = new ConcurrentHashMap<>();
    private List<String> dbShadeKeys = Collections.synchronizedList(new ArrayList<>());
    private PerfTestFootMarker perfTestFootMarker;

    public PerfTestRoutingDataSource(DS originalDataSource, Environment environment, PerfTestFootMarker perfTestFootMarker) {
        this.originalDataSource = originalDataSource;
        this.environment = environment;
        this.perfTestFootMarker = perfTestFootMarker;
    }

    @Override
    public void afterPropertiesSet() {
        if(thisInitted){
            return;
        }
        thisInitted = true;

        targetDataSources.put(ORIGINAL_DATASOURCE, originalDataSource);
        String originalUrl = getJdbcUrl(originalDataSource);

        originalScheme = getOriginalScheme(originalUrl);

        super.setTargetDataSources(targetDataSources);
        initShadeDbMap();
        super.afterPropertiesSet();
    }

    private void initShadeDbMap() {
        LinkedHashMap<String, Object> properties = SpringEnvironmentUtils.getFlatEnvironments(environment);
        properties.entrySet().stream().forEach(x -> {
            if (x.getKey().endsWith(SHADE_SCHEMA)) {
                String dbUrl = environment.getProperty(x.getKey().replace(SHADE_SCHEMA, ""));
                if (getJdbcUrl(originalDataSource).equals(dbUrl)) {//是当前数据源的影子配置时才加入
                    normalDbUrl2shadeUrlMap.put(dbUrl, x.getValue().toString());
                    if (!dbShadeKeys.contains(x.getKey())) {
                        dbShadeKeys.add(x.getKey());
                    }
                }
            }
        });
    }

    protected abstract String getJdbcUrl(DS originalDataSource);

    /**
     * 根据原始数据源得到对应的影子DataSource，影子数据库不存在则返回null.
     * 按照约定，影子库url和原始库url一致，只是scheme部分加入前缀perf__,用户名密码也必须相同，同一个用户要能看到这两个库
     * @param originalDataSource
     */
    private DS getTestDataSource(DS originalDataSource){
        String testUrl = normalDbUrl2shadeUrlMap.get(getJdbcUrl(originalDataSource));//如果存在影子库单独配置则优先使用该配置

        if(testUrl == null && !StringUtils.isBlank(originalScheme)) {
            Set<String> schemes = new HashSet<>();
            //用show databases命令检测所有数据库scheme，目前只支持mysql数据库的检测
            try (
                Connection connection = originalDataSource.getConnection();
                Statement statement = connection.createStatement();
                ResultSet rs = statement.executeQuery("show databases");
            ){
                while (rs.next()) {
                    schemes.add(rs.getString(1));
                }
            } catch (SQLException e) {
                throw new RuntimeException(e);
            }
            log.info("schemes of originalScheme:{}", schemes);
            String testScheme = TEST_SCHEME_PREFIX + originalScheme;
            if(schemes.contains(testScheme)){//影子库存在
                testUrl = getTestUrl(getJdbcUrl(originalDataSource));
            }
        }

        if(testUrl == null){//影子库不存在
            return null;
        }

        DS copyedDataSource = copyDataSource(testUrl, originalDataSource);

        return copyedDataSource;
    }

    protected abstract DS copyDataSource(String testUrl, DS originalDataSource);

    /**
     * 根据原始的数据源url得到对应的影子库url，按照约定，影子库url和原始库url一致，只是scheme部分加入前缀perf__
     * @param originalUrl
     * @return
     */
    private String getTestUrl(String originalUrl){
        int index = originalUrl.lastIndexOf('/');
        if(index > 0){
            return originalUrl.substring(0, index + 1) + TEST_SCHEME_PREFIX + originalUrl.substring(index+1, originalUrl.length());
        }else{
            throw new IllegalArgumentException(originalUrl + " is invalid");
        }
    }

    /**
     * 根据原始的数据源url得到它的scheme
     * @param originalUrl
     * @return
     */
    private String getOriginalScheme(String originalUrl){
        if(originalUrl.startsWith("jdbc:phoenix")){//hbase phoenix没有scheme，只支持设置.shade
            return null;
        }

        int index = originalUrl.lastIndexOf('/');
        if(index > 0){
            String schemeAfter = originalUrl.substring(index+1, originalUrl.length());

            int index1 = schemeAfter.indexOf('?');
            if(index1 > 0){
                return schemeAfter.substring(0, index1);
            }else{
                return schemeAfter;
            }
        }else{
            throw new IllegalArgumentException(originalUrl + " is invalid");
        }
    }

    @Override
    protected DataSource determineTargetDataSource() {
        Assert.notNull(this.targetDataSources, "DataSource router not initialized");
        Object lookupKey = determineCurrentLookupKey();
        DataSource dataSource = (DataSource)this.targetDataSources.get(lookupKey);

        if (dataSource == null) {
            throw new IllegalStateException("Cannot determine target DataSource for lookup key [" + lookupKey + "]");
        }
        return dataSource;
    }

    /**
     * 调用refresh后配置变更,重新初始化测试库
     */
//    @EventListener(EnvironmentChangeEvent.class)//这个注解不会生效，当前方法会被ext的其他地方调用
    public void onEnvironmentChange(EnvironmentChangeEvent event) {
        if(event.getKeys().stream().filter(s -> s.endsWith(SHADE_SCHEMA)).count() > 0){
            initShadeDbMap();
            if (testDataSourceInitted) {
                testDataSourceInitted = false;
            }
        } else if(testDataSourceInitted && !hasTestDataSource){
            //刷新后如果发现刷新前没有找到影子库，则重置标记, 让后续重新寻找影子库是否存在.
            testDataSourceInitted = false;
        }
    }

    protected Object determineCurrentLookupKey(){
        if(PerfTestContext.isCurrentInPerfTestMode()){//压测时，就算压测影子库不存在也不走原始库，防止产生脏数据
            if(!testDataSourceInitted){//在这里延迟加载
                initTestDataSource();
            }

            if(!hasTestDataSource){
                //压测时找不到影子库的异常抛出HystrixBadRequestException，以防触发客户端熔断
                throw new HystrixBadRequestException("", new IllegalStateException(getJdbcUrl(originalDataSource) + "的影子库不存在，放弃本次请求（如果在创建影子库后仍然遇到此异常，请在棱镜进行【刷新配置】操作后再试）"));
            }
            PerfTestContext.debugInfo("PerfTest DB");

            perfTestFootMarker.markDb(getJdbcUrl(shadeDataSource));
            return TEST_DATASOURCE;
        }

        return ORIGINAL_DATASOURCE;
    }

    /**
     * 初始化影子库的DataSource，如果找不到影子库，则为null
     */
    private void initTestDataSource(){
        synchronized (this){
            if(!testDataSourceInitted) {
                DS testDataSource = getTestDataSource(originalDataSource);

                if (testDataSource != null) {
                    hasTestDataSource = true;
                    shadeDataSource = testDataSource;
                    targetDataSources.put(TEST_DATASOURCE, testDataSource);//替换
                    log.info("检测到对数据库：{}的压测行为，已初始化压测专用数据库连接池", originalScheme);
                } else {
                    logger.warn(getJdbcUrl(originalDataSource) + " 没有对应的影子数据库（在同个库中以perf__开头的scheme, 或者添加.shade配置），如果你配置了，需要【刷新配置】生效");
                }

                testDataSourceInitted = true;
            }
        }
    }

    @Override
    public synchronized void destroy() throws Exception {
        if(targetDataSources != null){
            for(Object object : targetDataSources.values()){
                try {
                    if (object instanceof AutoCloseable) {
                        ((AutoCloseable) object).close();
                    } else {
                        log.warn("[notifyme]DataSourceClass:{} 没有实现AutoCloseable，无法关闭, 需要ext实现之", object.getClass().toString());
                    }
                }catch(Exception e){
                    // ignore
                }
            }
        }
    }

    public DataSource getOriginalDataSource() {
        return originalDataSource;
    }

    public DataSource getShadeDataSource() {
        return shadeDataSource;
    }

}
