package cn.com.duibaboot.ext.autoconfigure.web;

import cn.com.duibaboot.ext.autoconfigure.web.cookie.Rfc2109CookieFixFilter;
import org.apache.catalina.connector.ClientAbortException;
import org.apache.catalina.connector.Request;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.web.server.WebFilter;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import javax.servlet.DispatcherType;
import javax.servlet.Filter;
import javax.servlet.Servlet;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;

/**
 * 给web应用自动配置监控url，访问http://localhost:${port}/monitor/check即可识别出服务器状态是否正常(OK表示正常，INVALID表示服务关闭),
 * 访问http://localhost:${port}/monitor/stop可以让check接口返回INVALID,
 * 访问http://localhost:${port}/monitor/start可以让check接口返回OK.
 * 其中start和stop接口限制了只能通过本机访问。
 * 这三个接口供给nginx或slb等负载均衡器做健康检查、优雅停机用，当应用启动，默认check接口返回OK，负载均衡把这个机器作为后端服务器（负载均衡会每隔1s检查一次这个接口状态，如果状态不正常，则把这个机器从后端服务器列表暂时排除），
 * 当需要关闭应用时，发布脚本先调用stop接口，等待3s左右，让负载均衡有机会检测到应用已停用，然后发布脚本再开始真正停止应用。如果没有这个优雅停机的过程，在应用关闭的时候，负载均衡仍然会把请求转发给这台机器，导致部分用户看到报错页面。（目前Tengine可以设置请求某个后端服务器错误时自动请求其他服务器，但这会增加rt。）
 */
@Configuration
public class WebMvcMonitorAutoConfiguration {

    @Configuration
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    public static class BootMonitorCheckFilterConfigurationServlet {

        @Resource
        private List<FilterRegistrationBean> filters;

        /**
         * 检查是否有Filter的顺序太靠前，防止出现中文乱码
         */
        @PostConstruct
        public void init(){
            for(FilterRegistrationBean filter : filters){
                if(filter.getOrder() == Ordered.HIGHEST_PRECEDENCE){
                    throw new IllegalStateException("filter:["+filter.getFilter().getClass().getName()+"] 's order is too small, must bigger than Integer.MIN_VALUE(the order of OrderedCharacterEncodingFilter), if not, there will be chinese garbled");
                }
            }
        }

        /**
         * 自动加入Http Filter，修复tomcat 8.5.* 版本对RFC2109的cookie识别出来两边会带上引号的问题。
         * 修复tomcat 8.5.*版本对RFC2109的cookie处理的问题。RFC2109的cookie比如：
         * $Version="1"; _ac="eyJhaWQiOjE5NTI0LCJjaWQiOjUzMzUyMjk2OX0=";$Path="/";$Domain="duiba.com.cn"
         * <br/>
         * tomcat8.5.*版本在处理的时候会把_ac中的左右两个双引号也作为value的一部分，导致程序获取失败；tomcat8.0.*版本没有此问题。
         * 附上rfc2109的文档：https://tools.ietf.org/html/rfc2109#section-4.3.4
         */
        @Configuration
        @ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
                WebMvcConfigurer.class, Request.class})
        @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
        public static class Rfc2019CookieFilterConfiguration{
            @Bean
            public FilterRegistrationBean rfc2019CookieFilterConfigurer(){
                Rfc2109CookieFixFilter filter = new Rfc2109CookieFixFilter();
                FilterRegistrationBean registrationBean = new FilterRegistrationBean();
                registrationBean.setFilter(filter);
                List<String> urlPatterns=new ArrayList<>();
                urlPatterns.add("/*");//拦截路径，可以添加多个
                registrationBean.setUrlPatterns(urlPatterns);
                registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
                registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 10);//在CharacterFilter后面
                return registrationBean;
            }
        }

        /**
         * 自动加入Http Filter，拦截这个异常：org.apache.catalina.connector.ClientAbortException: java.io.IOException: 断开的管道
         * <br/>
         * 改为打印warn日志
         * 避免其上报给loginsight和cat监控。
         */
        @Configuration
        @ConditionalOnClass({ Servlet.class, DispatcherServlet.class,
                WebMvcConfigurer.class, ClientAbortException.class })
        @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
        public static class ClientAbortExceptionIgnoreFilterConfiguration{
            @Bean
            public FilterRegistrationBean clientAbortExceptionIgnoreFilterConfigurer(){
                ClientAbortExceptionIgnoreFilter filter = new ClientAbortExceptionIgnoreFilter();
                FilterRegistrationBean registrationBean = new FilterRegistrationBean();
                registrationBean.setFilter(filter);
                List<String> urlPatterns=new ArrayList<>();
                urlPatterns.add("/*");//拦截路径，可以添加多个
                registrationBean.setUrlPatterns(urlPatterns);
                registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
                registrationBean.setOrder(2);//在cat后面，防止不必要的异常让cat上报
                return registrationBean;
            }
        }

        /**
         * 为了防止健康检查URL被应用内的LoginFilter登录拦截器拦截而跳转302，由原来的直接使用Controller改为使用Filter
         */
        @Bean
        public FilterRegistrationBean bootMonitorCheckFilterConfigurer(){
            BootMonitorCheckFilter filter=new BootMonitorCheckFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<>();
            urlPatterns.add("/monitor/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(Ordered.HIGHEST_PRECEDENCE + 1);//最高优先级
            return registrationBean;
        }

        /**
         * 自动加入Http Filter，对超时请求打印warn日志,for servlet(备注：reactive版本暂时没有实现，不能用threadlocal来实现。)
         */
        @Bean
        public FilterRegistrationBean dbTimeProfileFilterConfigurer(){
            DBTimeProfileFilter filter=new DBTimeProfileFilter();
            FilterRegistrationBean registrationBean = new FilterRegistrationBean();
            registrationBean.setFilter(filter);
            List<String> urlPatterns=new ArrayList<>();
            urlPatterns.add("/*");//拦截路径，可以添加多个
            registrationBean.setUrlPatterns(urlPatterns);
            registrationBean.setDispatcherTypes(EnumSet.of(DispatcherType.REQUEST));
            registrationBean.setOrder(-1);
            return registrationBean;
        }

        @Bean
        public Filter featureFilter(){
            return new FeatureFilter();
        }
    }


    @Configuration
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.REACTIVE)
    public static class BootMonitorCheckFilterConfigurationReactive {
        @Bean
        public WebFilter bootMonitorCheckFilterConfigurer(){
            return new BootMonitorCheckReactiveFilter();
        }

        @Bean
        public WebFilter featureFilter(){
            return new FeatureReactiveFilter();
        }
    }
}
