package org.apache.http.pool;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duiba.boot.utils.AopTargetUtils;
import com.netflix.loadbalancer.Server;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.lang.StringUtils;
import org.apache.http.*;
import org.apache.http.client.HttpClient;
import org.apache.http.conn.ManagedHttpClientConnection;
import org.apache.http.conn.routing.HttpRoute;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.environment.EnvironmentChangeEvent;
import org.springframework.context.ConfigurableApplicationContext;
import org.springframework.context.event.EventListener;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;

//TODO 临时用，需要增强

/**
 * 最少活跃数负载均衡算法。
 * <br/>
 * 实际上是在发起调用前过滤掉当前活跃http连接数较高的服务实例，以让调用优先发给响应较快的机器.
 * <br/>
 * 通过加入配置 duiba.ribbon.loadbalance.type=leastactive 来激活,支持refresh动态调整
 */
@Component
//@ConditionalOnProperty(name="duiba.ribbon.loadbalance.type", havingValue = "leastactive")
public class LeastActiveServerListFilter implements RibbonServerListFilter {

    private static final Logger logger = LoggerFactory.getLogger(LeastActiveServerListFilter.class);

    private Map<HttpRoute, RouteSpecificPool<HttpRoute, ManagedHttpClientConnection, PoolEntry<HttpRoute, ManagedHttpClientConnection>>> routeToPool;

    private AtomicLong count = new AtomicLong();

    private static final String DUIBA_RIBBON_LOADBANCE_TYPE_KEY = "duiba.ribbon.loadbalance.type";

    private static final String DUIBA_RIBBON_LOADBANCE_TYPE_LEAST_ACTIVE = "leastactive";

    @Value("${"+DUIBA_RIBBON_LOADBANCE_TYPE_KEY+":}")
    private volatile String loadBalanceType;

    @EventListener(classes = EnvironmentChangeEvent.class)
    public void onEnvRefreshed(EnvironmentChangeEvent event){
        loadBalanceType = ((ConfigurableApplicationContext)event.getSource()).getEnvironment().getProperty(DUIBA_RIBBON_LOADBANCE_TYPE_KEY);
    }

    @Resource
    public void setApacheHttpClient(HttpClient apacheHttpClient) throws Exception{
        apacheHttpClient = AopTargetUtils.getTarget(apacheHttpClient);

        Field field = apacheHttpClient.getClass().getDeclaredField("connManager");
        field.setAccessible(true);
        PoolingHttpClientConnectionManager cm = (PoolingHttpClientConnectionManager)field.get(apacheHttpClient);

        field = cm.getClass().getDeclaredField("pool");
        field.setAccessible(true);
        AbstractConnPool pool = (AbstractConnPool)field.get(cm);

        field = AbstractConnPool.class.getDeclaredField("routeToPool");
        field.setAccessible(true);

        Map<HttpRoute, RouteSpecificPool<HttpRoute, ManagedHttpClientConnection, PoolEntry<HttpRoute, ManagedHttpClientConnection>>> routeToPool =
                (Map<HttpRoute, RouteSpecificPool<HttpRoute, ManagedHttpClientConnection, PoolEntry<HttpRoute, ManagedHttpClientConnection>>>)field.get(pool);

        this.routeToPool = routeToPool;
    }

    @Override
    public List<Server> filter(List<Server> serverList, Object key){
        if(!StringUtils.equals(loadBalanceType, DUIBA_RIBBON_LOADBANCE_TYPE_LEAST_ACTIVE)){
            return serverList;
        }

        if(serverList.isEmpty()
                || !(serverList.get(0) instanceof DiscoveryEnabledServer)){
            return serverList;
        }

        List<Server> servers = new ArrayList<>(serverList);

        List<Integer> leaseList = new ArrayList<>();
        for(Server server : servers){
            String host = ((DiscoveryEnabledServer)server).getInstanceInfo().getIPAddr();
            int port = ((DiscoveryEnabledServer)server).getInstanceInfo().getPort();
            RouteSpecificPool<HttpRoute, ManagedHttpClientConnection, PoolEntry<HttpRoute, ManagedHttpClientConnection>> val = routeToPool.get(new HttpRoute(new HttpHost(host, port, "http")));
            if(val != null){
//                System.out.println(server + "  ->  " + "leasedCount:" + val.getLeasedCount() + ", available:" + val.getAvailableCount()+",pend:"+val.getPendingCount());
                leaseList.add(val.getLeasedCount());
            }else{
                leaseList.add(0);
            }
        }

        int totalLeased = 0;
        for(Integer leased : leaseList){
            totalLeased += leased;
        }
//        if(totalLeased > 10){
//            if(count.incrementAndGet() % 100 == 0){//每100次掉用只打印1次.
//                logger.warn("{}", );
//            }
//        }
        float avgLeased = totalLeased * 1.0f / leaseList.size();
        for(int i=leaseList.size()-1;i>=0;i--){
            Integer leased = leaseList.get(i);
            if(leased > avgLeased && leased > 5) {
                servers.remove(i);
            }
        }

        if(servers.isEmpty()){//如果经过过滤后机器数为0，则返回原来的服务器列表
            return serverList;
        }

        return servers;
    }

}
