package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign;

import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.hystrix.HystrixFeignConfiguration;
import com.netflix.hystrix.HystrixCommand;
import feign.Contract;
import feign.RequestInterceptor;
import feign.hystrix.HystrixFeign;
import java.util.ArrayList;
import java.util.List;
import javax.annotation.Resource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.openfeign.AnnotatedParameterProcessor;
import org.springframework.cloud.openfeign.FeignFormatterRegistrar;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.core.convert.ConversionService;
import org.springframework.format.support.DefaultFormattingConversionService;
import org.springframework.format.support.FormattingConversionService;

/**
 * Feign客户端自定义，覆盖 FeignClientsConfiguration 中对应配置，并新增Feign的Interceptor以方便进行cat监控
 */
@Configuration
@ConditionalOnClass({ HystrixCommand.class, HystrixFeign.class })
@Import(HystrixFeignConfiguration.class)
@EnableConfigurationProperties(DuibaFeignProperties.class)
public class FeignAutoConfiguration {

    @Resource
    private DuibaFeignProperties duibaFeignProperties;

    @Autowired(required = false)
    private List<AnnotatedParameterProcessor> parameterProcessors = new ArrayList<>();

    @Autowired(required = false)
    private List<FeignFormatterRegistrar> feignFormatterRegistrars = new ArrayList<>();

    /**
     * 使用自定义的 Contract，对于AdvancedFeignClient进行特殊处理，自动扫描AdvancedFeignClient下的所有方法为FeignClient(使用方法名作为path)，并把所有参数使用json序列化
     * @param customFeignConversionService
     * @return
     */
    @Bean
    public Contract feignContract(ConversionService customFeignConversionService) {
        return new CustomSpringMvcContract(this.parameterProcessors, customFeignConversionService, duibaFeignProperties);
    }

    @Bean
    public FormattingConversionService customFeignConversionService() {
        FormattingConversionService conversionService = new DefaultFormattingConversionService();
        for (FeignFormatterRegistrar feignFormatterRegistrar : feignFormatterRegistrars) {
            feignFormatterRegistrar.registerFormatters(conversionService);
        }
        return conversionService;
    }

    /**
     * FeignClient请求拦截器，附加cat分布式跟踪id
     * @return
     */
    @Bean
    public RequestInterceptor getCustomRequestInterceptor(){
        return new CustomRequestInterceptor(duibaFeignProperties);
    }

    /**
     * 一个remote接口。
     * List<String> testList(List<String> paramList);
     * 如果paramList在客户端传的是 空数组，那么boot1和boot2会有以下两种表现
     * boot1: paramList = []
     * boot2: paramList = [null]
     *
     * 上面的情况，在feign构建httpRequest的时候
     * boot1构建的请求是：http://172.16.20.156:2222/remoteFlowReplayUserService/testList
     * boot2构建的请求是：http://172.16.20.156:2222/remoteFlowReplayUserService/testList?_p0
     *
     * 该拦截器用于将boot2中feign的该表现修改成和boot1中一致
     * @return
     */
    @Bean
    @ConditionalOnProperty(name = "duiba.feign.serialization", havingValue = "json", matchIfMissing = true)
    public RequestInterceptor getNewFeignJSONSerializationCompatibleCustomRequestInterceptor() {
        return new NewFeignJSONSerializationCompatibleCustomRequestInterceptor();
    }
    
}
