package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer;

import java.util.List;
import java.util.Random;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.Server;

/**
 * 加权随机 负载均衡规则
 *
 * @author wenqi.huang
 */
public class WeightedRandomRule extends FilterBasedRule {

    private final Random random = new Random();

    public WeightedRandomRule(ILoadBalancer lb, List<RibbonServerListFilter> serverListFilters) {
        super(serverListFilters);
        setLoadBalancer(lb);
    }

    @Override
    protected Server chooseFromServers(List<Server> servers, ILoadBalancer lb, Object key) {
        Server server = null;
        int count = 0;
        int maxFindTimes = Math.min(10, servers.size() + 1);
        while (server == null && count++ < maxFindTimes) {
            server = chooseRoundRobinWithTimeBasedWeight(servers);

            if (server == null) {
                /* Transient. */
                Thread.yield();
                continue;
            }

//            if (server.isAlive() && (server.isReadyToServe())) {
//                return (server);
//            }
//
//            // Next.
//            server = null;

            return server;
        }

        if (count >= 10) {
            log.warn("No available alive servers after 10 tries from load balancer: "
                    + lb);
        }
        return server;
    }

    /**
     * 使用权重加权随机算法获取一个服务器。其中权重根据服务器启动时间变化，以达到对于刚启动的服务器逐渐增加流量的目的
     * @param eligible
     * @return
     */
    private Server chooseRoundRobinWithTimeBasedWeight(List<Server> eligible){
        int length = eligible.size(); // 总个数
        int totalWeight = 0; // 总权重
        boolean sameWeight = true; // 权重是否都一样
        int[] weights = new int[length];
        for (int i = 0; i < length; i++) {
            int weight = getTimeBasedWeight(eligible.get(i));
            weights[i] = weight;
            totalWeight += weight; // 累计总权重
            if (sameWeight && i > 0
                    && weight != weights[i - 1]) {
                sameWeight = false; // 计算所有权重是否一样
            }
        }
        if (totalWeight > 0 && ! sameWeight) {
            // 如果权重不相同且权重大于0则按总权重数随机
            int offset = random.nextInt(totalWeight);
            // 并确定下一个值落在哪个片断上
            for (int i = 0; i < length; i++) {
                offset -= weights[i];
                if (offset < 0) {
                    return eligible.get(i);
                }
            }
        }

        // 如果权重相同或权重为0则直接随机即可
        return eligible.get(random.nextInt(length));
    }
}
