package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duiba.boot.flowreplay.FlowReplayCustomizeSpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayErrorMsgTypeEnum;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.StepDiffColumn;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.CustomizeFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.SpanType;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;

import java.lang.reflect.Method;
import java.util.Set;

/**
 * FlowReplayCustomizeSpan 注解切面，用于实际的回归
 * Created by guoyanfei .
 * 2019-05-23 .
 */
@Slf4j
@Aspect
public class ReplayCustomizeFlowReplaySpanPlugin {

    @Around("execution(* *.*(..)) && @annotation(cn.com.duiba.boot.flowreplay.FlowReplayCustomizeSpan)")
    public Object customizeJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable {  // NOSONAR
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        FlowReplayCustomizeSpan customizeSpanAnno = signature.getMethod().getAnnotation(FlowReplayCustomizeSpan.class);
        if (customizeSpanAnno == null) {
            return joinPoint.proceed();
        }
        // 当前不是回归的环境 或者 当前请求不是回归请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return joinPoint.proceed();
        }

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回归的span，用于回归详情的构建
        CustomizeFlowReplaySpan replayDetailSpan = createReplayDetailSpan(signature, joinPoint.getArgs(), span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.CUSTOMIZE != span.getSpanType()) {
            String expert = SpanType.CUSTOMIZE.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        CustomizeFlowReplaySpan customizeSpan = (CustomizeFlowReplaySpan) span;

        Method method = signature.getMethod();
        Object[] parameterValues = joinPoint.getArgs();

        // 方法名是否相同
        if (!FlowReplayUtils.isMethodEqual(customizeSpan.getMethodName(), method)) {
            String expert = customizeSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_701, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        // 参数类型是否相同
        if (!FlowReplayUtils.isArgumentsTypesEqual(customizeSpan.getParameterTypes(), method.getParameterTypes())) {
            String expert = FlowReplayUtils.stringArrayToString(customizeSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(method.getParameterTypes());
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_702, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            // 对比 [录制的参数列表] 和 [回归的参数列表] ，记录对比过程中不相同的参数
            Set<StepDiffColumn> stepDiffColumns = FlowReplayUtils.compareArray(ReplayTraceContext.getContextTraceId(), ReplayTraceContext.getCurrentSpanIdx(), customizeSpan.getParameterValues(), parameterValues);
            ReplayTraceContext.addAllStepDiffColumns(stepDiffColumns);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_706, e);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        // 返回值类型是否相同
        if (!FlowReplayUtils.isReturnTypeEqual(customizeSpan.getReturnType(), method.getReturnType())) {
            String expert = customizeSpan.getReturnType();
            String actual = method.getReturnType() != null ? method.getReturnType().getName() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_704, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        // 方法所在类全路径是否相同
        if (!FlowReplayUtils.isTypeFullPathEqual(customizeSpan.getTypeFullPath(), signature.getDeclaringTypeName())) {
            String expert = customizeSpan.getTypeFullPath();
            String actual = signature.getDeclaringTypeName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_705, expert, actual);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            // 都相同mock返回值
            Object mockRet = customizeSpan.getReturnValue();

            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回归详情用的span中
            replayDetailSpan.setReturnValue(mockRet);
            return mockRet;
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_707, e);
            throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    private CustomizeFlowReplaySpan createReplayDetailSpan(MethodSignature signature, Object[] parameterValues, FlowReplaySpan span) {
        CustomizeFlowReplaySpan replayDetailSpan = CustomizeFlowReplaySpan.createSpan(signature, parameterValues, null);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }
}
