package cn.com.duibaboot.ext.autoconfigure.mybatis;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import org.apache.ibatis.io.VFS;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * mybatis自动配置，目前用于修复mybatis在spring下的一个问题：https://github.com/mybatis/mybatis-3/issues/325
 * 这个问题大概表现是：springboot打成jar包后，运行时，mybatis会扫描typeAliasesPackage配置下的类，由于这个类在springboot jar内部的嵌入jar中，导致找不到该类。
 *
 * Created by wenqi.huang on 2016/11/22.
 */
@Configuration
public class MybatisAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(MybatisAutoConfiguration.class);

    @Configuration
    @ConditionalOnClass({SqlSessionTemplate.class, SqlSessionFactoryBean.class, SqlSessionFactory.class, VFS.class})
    public static class MyBatisBugfixPostProcessorConfiguration{

        /**
         * 声明后置处理器，spring全部bean初始化完成前调用，给所有 SqlSessionFactoryBean 注入自定义的VFS，修复bug
         * @return
         */
        //注释了bean了方法名不能冲突，否则只有一个会被应用。。坑爹
        @Bean
        public static SpecifiedBeanPostProcessor mybatisBugfixPostProcessorConfigurer(){
            //mybatis-spring 1.3.0才有getVfs/setVfs方法，所以需要判断是否有此方法
            boolean hasVfsMethod = false;
            try {
                SqlSessionFactoryBean.class.getMethod("getVfs");
                hasVfsMethod = true;
            } catch (NoSuchMethodException e) {
                logger.warn("method[org.mybatis.spring.SqlSessionFactoryBean.getVfs()] is not exist, 无法自动修复mybatis的bug，请升级mybatis-spring到1.3.1或以上版本(否则cat有可能无法对sql进行监控)");
            }

            final boolean hasVfsMethodFinal = hasVfsMethod;

            return new SpecifiedBeanPostProcessor<SqlSessionFactoryBean>() {

                @Override
                public int getOrder() {
                    return -1;
                }

                @Override
                public Class<SqlSessionFactoryBean> getBeanType() {
                    return SqlSessionFactoryBean.class;
                }

                @Override
                public Object postProcessBeforeInitialization(SqlSessionFactoryBean bean, String beanName) throws BeansException {
                    if(hasVfsMethodFinal) {
                        if (bean.getVfs() == null) {
                            bean.setVfs(SpringBootVFS.class);
                        }
                    }

                    return bean;
                }

                @Override
                public Object postProcessAfterInitialization(SqlSessionFactoryBean bean, String beanName) throws BeansException {
                    return bean;
                }
            };

        }
    }
}
