package cn.com.duibaboot.ext.autoconfigure.datasource;

import cn.com.duibaboot.ext.autoconfigure.perftest.PerfTestRoutingDataSource;
import com.zaxxer.hikari.HikariDataSource;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.jdbc.DatabaseDriver;
import org.springframework.util.StringUtils;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

//duiba.datasource
@ConfigurationProperties(DuibaDataSourceProperties.DUIBA_DATA_SOURCE_PROPERTIES_PREFIX)
public class DuibaDataSourceProperties {

    public static final String DUIBA_DATA_SOURCE_PROPERTIES_PREFIX = "duiba";

    //duiba.datasource
    private Map<String, DataSourcePropertiesInner> datasource = new LinkedHashMap<>();

    public Map<String, DataSourcePropertiesInner> getDatasource() {
        return datasource;
    }

    public void setDatasource(Map<String, DataSourcePropertiesInner> datasource) {
        this.datasource = datasource;
    }

    public static class DataSourcePropertiesInner{

        private static final Logger logger = LoggerFactory.getLogger(DataSourcePropertiesInner.class);
        private static final Class<?> HIKARI_CP_CLASS;

        static{
            Class<?> hikariCpClass = null;
            try {
                hikariCpClass = Class.forName("com.zaxxer.hikari.HikariDataSource");
            } catch (ClassNotFoundException e) {
                //Ignore
            }

            HIKARI_CP_CLASS = hikariCpClass;
        }

        /**
         * JDBC url of the database.
         */
        private String url;

        /**
         * Login user of the database.
         */
        private String username;

        /**
         * Login password of the database.
         */
        private String password;

        /**
         * 连接池最大数量
         */
        private Integer maxTotal = 8;

        /**
         * 连接池初始值
         */
        @Deprecated
        private Integer initialSize = 3;

        /**
         * 连接池最大空闲
         */
        @Deprecated
        private Integer maxIdle = 8;

        /**
         * 连接池最小空闲
         */
        private Integer minIdle = 0;

        /**
         * 等待连接池的最大时间（ms）,不能小于250ms，如果小于250ms，会被校正到250ms
         */
        private Integer maxWaitMillis = 3000;

        private List<String> connectionInitSqls = new ArrayList<>();

        public String getUrl() {
            return url;
        }

        public void setUrl(String url) {
            this.url = url;
        }

        public String getUsername() {
            return username;
        }

        public void setUsername(String username) {
            this.username = username;
        }

        public String getPassword() {
            return password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public Integer getMaxTotal() {
            return maxTotal;
        }

        public void setMaxTotal(Integer maxTotal) {
            this.maxTotal = maxTotal;
        }

        public Integer getInitialSize() {
            return initialSize;
        }

        public void setInitialSize(Integer initialSize) {
            this.initialSize = initialSize;
        }

        public Integer getMaxIdle() {
            return maxIdle;
        }

        public void setMaxIdle(Integer maxIdle) {
            this.maxIdle = maxIdle;
        }

        public Integer getMinIdle() {
            return minIdle;
        }

        public void setMinIdle(Integer minIdle) {
            this.minIdle = minIdle;
        }

        public Integer getMaxWaitMillis() {
            return maxWaitMillis;
        }

        public void setMaxWaitMillis(Integer maxWaitMillis) {
            this.maxWaitMillis = maxWaitMillis;
        }

        public Class<? extends DataSource> getType() {
            try {
                Class.forName("com.zaxxer.hikari.HikariDataSource");
            } catch (ClassNotFoundException e) {
                throw new IllegalStateException("检测到你的项目中需要使用dataSource，目前强制使用HikariCP，请引入HikariCP 的jar依赖.");
            }
            return HikariDataSource.class;
        }


        /**
         * Determine the driver to use based on this configuration and the environment.
         * @return the driver to use
         * @since 1.4.0
         */
        public String determineDriverClassName() {
//            if (StringUtils.hasText(this.driverClassName)) {
//                Assert.state(driverClassIsLoadable(),
//                        "Cannot load driver class: " + this.driverClassName);
//                return this.driverClassName;
//            }
            String driverClassName = null;

            if (StringUtils.hasText(this.url)) {
                if(url.startsWith("jdbc:mysql:")){
                    driverClassName = "com.mysql.cj.jdbc.Driver";
                }else {
                    driverClassName = DatabaseDriver.fromJdbcUrl(this.url).getDriverClassName();
                    if (driverClassName == null) {
                        if (this.url.startsWith("jdbc:phoenix:")) {//phoenix
                            driverClassName = "org.apache.phoenix.jdbc.PhoenixDriver";
                        }
                    }
                }
            }

            if (!StringUtils.hasText(driverClassName)) {
                throw new IllegalStateException("can't determine driverClassName of url:" + url);
            }
            return driverClassName;
        }

        /**
         * Initialize a {@link BeanDefinitionBuilder} with the state of this instance.
         * @return a {@link BeanDefinitionBuilder} initialized with the customizations defined on
         * this instance
         */
        public BeanDefinitionBuilder initializeDataSourceBeanDefinitionBuilder(String dataSourceName) {
            if(StringUtils.isEmpty(url)){
                return null;
            }
            Class dataSourceClass = getType();
            if(HIKARI_CP_CLASS != null && HIKARI_CP_CLASS.isAssignableFrom(dataSourceClass)){
                if(maxWaitMillis < 260){
                    maxWaitMillis = 260;
                }
                int validationTimeout = Math.min(5000, Math.max(250, maxWaitMillis - 10));//validationTimeout必须比connectionTimeout小

                BeanDefinitionBuilder builder = BeanDefinitionBuilder.genericBeanDefinition(dataSourceClass)
                        .addPropertyValue("driverClassName", determineDriverClassName())
                        .addPropertyValue("jdbcUrl", getUrl())
                        .addPropertyValue("maximumPoolSize", maxTotal)
                        .addPropertyValue("poolName", dataSourceName)
                        .addPropertyValue("minimumIdle", minIdle)//官方不建议设置minimumIdle
                        .addPropertyValue("connectionTimeout", maxWaitMillis)
                        .addPropertyValue("validationTimeout", validationTimeout)//validationTimeout必须比connectionTimeout小
                        .setDestroyMethodName("close");

                if(!org.apache.commons.lang3.StringUtils.isBlank(getUsername())) {
                    builder.addPropertyValue("username", getUsername());
                }
                if(!org.apache.commons.lang3.StringUtils.isBlank(getPassword())) {
                    builder.addPropertyValue("password", getPassword());
                }
                if(connectionInitSqls.size() == 1) {
                    builder.addPropertyValue("connectionInitSql", connectionInitSqls.get(0));
                }else if(connectionInitSqls.size() > 1){
                    throw new IllegalStateException("connectionInitSqls只支持设置一个sql");
                }

                return builder;
            }

            throw new IllegalStateException("unsupportted dataSource type:" + dataSourceClass.getName() + ", currently only HikariCP is supported");
        }

        /**
         * 运行时修改数据源配置
         * @param dataSourceBeanName spring bean id
         * @param dataSource 需要运行时修改配置的数据源
         */
        public void bindProperties(String dataSourceBeanName, DataSource dataSource){
            List<DataSource> dataSourcesNeedChange = new ArrayList<>();
            if(dataSource instanceof PerfTestRoutingDataSource){
                PerfTestRoutingDataSource ds1 = (PerfTestRoutingDataSource)dataSource;
                dataSourcesNeedChange.add(ds1.getOriginalDataSource());
                if(ds1.getShadeDataSource() != null) {
                    dataSourcesNeedChange.add(ds1.getShadeDataSource());
                }
            }else{
                dataSourcesNeedChange.add(dataSource);
            }

            for(DataSource ds : dataSourcesNeedChange){
                if(HIKARI_CP_CLASS != null && HIKARI_CP_CLASS.isAssignableFrom(ds.getClass())){
                    if(maxWaitMillis < 260){
                        maxWaitMillis = 260;
                    }
                    int validationTimeout = Math.min(5000, Math.max(250, maxWaitMillis - 10));//validationTimeout必须比connectionTimeout小

                    HikariDataSource d = (HikariDataSource)ds;
                    // jdbcUrl 不能实时刷新
//                    d.setJdbcUrl(getUrl());
//                    d.setUsername(getUsername());
//                    d.setPassword(getPassword());
                    d.setMaximumPoolSize(maxTotal);
                    d.setMinimumIdle(minIdle);//官方不建议设置minimumIdle
                    d.setConnectionTimeout(maxWaitMillis);
                    d.setValidationTimeout(validationTimeout);//validationTimeout必须比connectionTimeout小
                }else{
                    logger.warn("不能刷新数据源类型：{}, beanId:{}，当前只支持刷新HikariCP数据源", ds.getClass().getName(), dataSourceBeanName);
                }
            }
        }

        public List<String> getConnectionInitSqls() {
            return connectionInitSqls;
        }

        public void setConnectionInitSqls(List<String> connectionInitSqls) {
            this.connectionInitSqls = connectionInitSqls;
        }
    }

}
