package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.RedisFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;

/**
 * 录制 RedisAtomicClient 的方法
 * Created by guoyanfei .
 * 2019-06-20 .
 */
@Slf4j
public class RecordRedisAtomicClientMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以录制
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canRecord(Method method, Object[] allArguments) {
        if (!FlowReplayTrace.isTraced()) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        if (!canRecord(method, allArguments)) {
            return;
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return;
        }
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(obj, method.getName(), allArguments);
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        if (!canRecord(method, allArguments)) {
            return ret;
        }
        if (!IgnoreSubInvokesContext.isInstMarked(obj, method.getName(), allArguments)) {
            return ret;
        }
        try {
            Object recordRet = null;
            if ("getLock".equals(method.getName())) {
                recordRet = (ret != null);
            } else {
                recordRet = ret;
            }
            RedisFlowReplaySpan span = RedisFlowReplaySpan.createSpan(method, allArguments, argumentsTypes, recordRet);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
        } catch (Throwable t) {
            log.error("RedisAtomicClient录制异常", t);
            // 录制异常，这个用例不录了
            FlowReplayTrace.remove();
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        // 如果捕捉到异常，并且正在录制中，那么不录这个用例
        if (FlowReplayTrace.isTraced()) {
            FlowReplayTrace.remove();
        }
    }
}
