package cn.com.duibaboot.ext.autoconfigure.grouping.dubbo;

import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.registry.NotifyListener;
import org.apache.dubbo.registry.Registry;
import org.apache.dubbo.registry.RegistryFactory;

import java.util.List;

/**
 * dubbo服务分组调用 注册时候的代理
 * 多场景的机器，在容器启动的时候，容器平台会在环境变量中加入 key: DUIBA_SERVICE_GROUP_KEY, value: 【服务分组key】 的环境变量。
 * 这里会在dubbo服务端注册的时候，把环境变量中的【服务分组key】取出来，加入到 准备注册的URL中，在服务调用的时候，如果请求中有服务分组参数，那么优先调用URL中有该参数的服务
 * Created by gyf .
 * 2020/11/12 .
 */
@Slf4j
public class ServiceGroupDubboRegistryFactoryWrapper implements RegistryFactory {

    public static final String SERVICE_GROUP_KEY = "serviceGroupKey";

    private final RegistryFactory registryFactory;

    public ServiceGroupDubboRegistryFactoryWrapper(RegistryFactory registryFactory) {
        this.registryFactory = registryFactory;
    }

    @Override
    public Registry getRegistry(URL url) {
        return new RegistryWrapper(this.registryFactory.getRegistry(url));
    }

    private static class RegistryWrapper implements Registry {

        private final Registry originRegistry;

        private URL appendProviderServiceGroupKey(URL url) {
            String side = url.getParameter("side");
            if ("provider".equals(side)) {
                String serviceGroupKey = System.getenv(ServiceGroupUtils.ENV_DUIBA_SERVICE_GROUP_KEY);
                // 如果是本地启用的话，修改duibaServiceGroupKey为 _duibaServiceGroupKey=dev_ip_本机ip
                if (StringUtils.isBlank(serviceGroupKey) && !SpringBootUtils.isJarInJarMode()) {
                    serviceGroupKey = ServiceGroupUtils.DUIBA_SERVICE_GROUP_IP_PREFIX + NetUtils.getLocalIp();
                }
                if (StringUtils.isNotBlank(serviceGroupKey)){
                    url = url.addParameter(SERVICE_GROUP_KEY, serviceGroupKey); //注册dubbo接口时添加参数，表示本机器是多场景的机器
                    log.warn("本服务为【服务分组（多场景）】的服务，环境变量中的场景key为: {}，该场景key已添加到dubbo提供者中。", serviceGroupKey);
                }
            }
            return url;
        }

        public RegistryWrapper(Registry originRegistry) {
            this.originRegistry = originRegistry;
        }

        @Override
        public URL getUrl() {
            return this.originRegistry.getUrl();
        }

        @Override
        public boolean isAvailable() {
            return this.originRegistry.isAvailable();
        }

        @Override
        public void destroy() {
            this.originRegistry.destroy();
        }

        @Override
        public void register(URL url) {
            this.originRegistry.register(this.appendProviderServiceGroupKey(url));
        }

        @Override
        public void unregister(URL url) {
            this.originRegistry.unregister(this.appendProviderServiceGroupKey(url));
        }

        @Override
        public void subscribe(URL url, NotifyListener listener) {
            this.originRegistry.subscribe(url, listener);
        }

        @Override
        public void unsubscribe(URL url, NotifyListener listener) {
            this.originRegistry.unsubscribe(url, listener);
        }

        @Override
        public List<URL> lookup(URL url) {
            return this.originRegistry.lookup(this.appendProviderServiceGroupKey(url));
        }
    }
}
