package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.Server;

import java.util.List;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicLong;

/**
 * [平滑]加权轮询 负载均衡规则
 *
 * 参考dubbo2.7中最新的[平滑]加权轮询算法
 * 参考文档：https://github.com/apache/dubbo/pull/2647
 * @see org.apache.dubbo.rpc.cluster.loadbalance.RoundRobinLoadBalance
 *
 * @author wenqi.huang
 */
public class WeightedRoundRobinRule extends FilterBasedRule {

    // 服务下线60s后，在当前实例维护的实例副本中，移除下线服务的实时信息
    private static final int RECYCLE_PERIOD = 60000;

    // 存放全实例副本当前的实时信息
    private ConcurrentMap<String, WeightedRoundRobin> weightMap = new ConcurrentHashMap<>();

//    private AtomicBoolean updateLock = new AtomicBoolean();

    public WeightedRoundRobinRule(ILoadBalancer lb, List<RibbonServerListFilter> serverListFilters) {
        super(serverListFilters);
        setLoadBalancer(lb);
    }

    @Override
    protected Server chooseFromServers(List<Server> servers, ILoadBalancer lb, Object key) {
        Server server = null;
        int count = 0;
        int maxFindTimes = Math.min(10, servers.size() + 1);
        while (server == null && count++ < maxFindTimes) {
            server = chooseSmoothRoundRobinWithTimeBasedWeight(servers);

            if (server == null) {
                /* Transient. */
                Thread.yield();
                continue;
            }

//            if (server.isAlive() && (server.isReadyToServe())) {
//                return (server);
//            }
//
//            // Next.
//            server = null;
            return server;
        }

        if (count >= 10) {
            log.warn("No available alive servers after 10 tries from load balancer: "
                    + lb);
        }
        return server;
    }

    /**
     * 使用[平滑]加权轮询算法获取一个服务器。
     * <p>
     * 其中权重根据服务器启动时间变化，以达到对于刚启动的服务器逐渐增加流量的目的。
     *
     * @param eligible 应用可用实例列表
     * @return
     */
    private Server chooseSmoothRoundRobinWithTimeBasedWeight(List<Server> eligible) {
        int totalWeight = 0;
        long maxCurrent = Long.MIN_VALUE;
        long now = System.currentTimeMillis();
        Server selectedServer = null;
        WeightedRoundRobin selectedWRR = null;
        for (Server server : eligible) {
            String serverId = server.getId();
            int weight = getTimeBasedWeight(server);

//            WeightedRoundRobin weightedRoundRobin = weightMap.get(ipAddr);
//            if (weightedRoundRobin == null) {
//                weightedRoundRobin = new WeightedRoundRobin();
//                weightedRoundRobin.setWeight(weight);
//                weightMap.putIfAbsent(ipAddr, weightedRoundRobin);
//            }

            WeightedRoundRobin weightedRoundRobin = weightMap.computeIfAbsent(serverId, k -> {
                WeightedRoundRobin wrr = new WeightedRoundRobin();
                wrr.setWeight(weight);
                return wrr;
            });
            if (weight != weightedRoundRobin.getWeight()) {
                //weight changed
                weightedRoundRobin.setWeight(weight);
            }
            long cur = weightedRoundRobin.increaseCurrent();
            weightedRoundRobin.setLastUpdate(now);
            if (cur > maxCurrent) {
                maxCurrent = cur;
                selectedServer = server;
                selectedWRR = weightedRoundRobin;
            }
            totalWeight += weight;
        }
//        if (!updateLock.get() && eligible.size() != weightMap.size()) {
//            if (updateLock.compareAndSet(false, true)) {
//                try {
//                    // copy -> modify -> update reference
//                    weightMap.entrySet().removeIf(item -> now - item.getValue().getLastUpdate() > RECYCLE_PERIOD);
//                } finally {
//                    updateLock.set(false);
//                }
//            }
//        }
        if (eligible.size() != weightMap.size()) {
            weightMap.entrySet().removeIf(item -> now - item.getValue().getLastUpdate() > RECYCLE_PERIOD);
        }
        if (selectedServer != null) {
            selectedWRR.sel(totalWeight);
            return selectedServer;
        }
        // 理论不会到这里
        // 如果代码bug走到这里，就随机选择服务器
        return eligible.get(ThreadLocalRandom.current().nextInt(eligible.size()));
    }

    protected static class WeightedRoundRobin {
        private int weight;
        private AtomicLong current = new AtomicLong(0);
        private long lastUpdate;

        public int getWeight() {
            return weight;
        }

        public void setWeight(int weight) {
            this.weight = weight;
            current.set(0);
        }

        public long increaseCurrent() {
            return current.addAndGet(weight);
        }

        public void sel(int total) {
            current.addAndGet(-1 * total);
        }

        public long getLastUpdate() {
            return lastUpdate;
        }

        public void setLastUpdate(long lastUpdate) {
            this.lastUpdate = lastUpdate;
        }
    }

}
