package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon.loadbalancer;

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duiba.wolf.utils.NumberUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import com.netflix.loadbalancer.ILoadBalancer;
import com.netflix.loadbalancer.Server;
import com.netflix.loadbalancer.ZoneAvoidanceRule;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.collections.CollectionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.ArrayList;
import java.util.List;

public abstract class FilterBasedRule extends ZoneAvoidanceRule {

	protected Logger log = LoggerFactory.getLogger(getClass());

	private List<RibbonServerListFilter> serverListFilters;

	public FilterBasedRule(List<RibbonServerListFilter> serverListFilters){
	    if (serverListFilters != null) {
            this.serverListFilters = new ArrayList<>(serverListFilters);
        }
	}

	@Override
	public Server choose(Object key) {
		ILoadBalancer lb = getLoadBalancer();
		if (lb == null) {
			log.warn("no load balancer");
			return null;
		}
		//注意这里调用的是getReachableServers方法，只会拿到isAlive的服务器列表进行选择
		List<Server> servers = getPredicate().getEligibleServers(lb.getReachableServers(), key);

		if(serverListFilters != null) {
			for (RibbonServerListFilter filter : serverListFilters) {
				servers = filter.filter(servers, key);
			}
		}

		if(servers.isEmpty()){
			log.info("No available alive server from lb.getReachableServers(), failback to lb.getAllServers()");
			//如果找不到，从eureka拿到的所有UP状态的服务器中找一台
			servers = lb.getAllServers();
		}

		if(CollectionUtils.isEmpty(servers)) {
			return null;
		}

		return chooseFromServers(servers, getLoadBalancer(), key);
	}

	protected abstract Server chooseFromServers(List<Server> servers, ILoadBalancer loadBalancer, Object key);

	/**
	 * 根据服务器启动时间计算服务器权重，启动时间越久，权重越大(预热时间5分钟，5分钟后weight会达到服务器设定的最大值)
	 * @param server
	 * @return
	 */
	protected int getTimeBasedWeight(Server server) {
		int weight = 100;
		if(server instanceof DiscoveryEnabledServer){
			String weightStr = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.WEIGHT_KEY);
			weight = NumberUtils.parseInt(weightStr, 100);
			weight = Math.max(weight, 0);
		}

		if (weight > 0) {
			//获取目标服务器启动时间
			long timestamp = 0;
			int warmUpTimeMillis = 300000;//预热时间，单位:ms
			if(server instanceof DiscoveryEnabledServer){
				String tsStr = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.SERVER_START_UP_TIME_KEY);
				timestamp = NumberUtils.parseLong(tsStr,0);
				timestamp = Math.max(timestamp, 0);

				String wuStr = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_WARMUP_TIMEMILLIS);
				warmUpTimeMillis = NumberUtils.parseInt(wuStr, 300000);
				warmUpTimeMillis = Math.max(warmUpTimeMillis, 0);//最少也要0秒预热时间
			}

			if (timestamp > 0L) {
				long uptime = System.currentTimeMillis() - timestamp;
				if (uptime >= Integer.MAX_VALUE) { //启动时间很久了，直接使用原权重，否则uptime后续转为int时会溢出变成负数（使用int是因为int计算更高效）。
				}
				else if (uptime <= 0) {
					weight = 1;
				}
				else if (uptime < warmUpTimeMillis) {
					weight = calculateWarmupWeight((int)uptime, warmUpTimeMillis, weight);
				}
			}
		}
		return weight;
	}

	/**
	 * 预热期间权重计算
	 *
	 * @param uptime
	 * @param warmup
	 * @param weight
	 * @return
	 */
	private int calculateWarmupWeight(int uptime, int warmup, int weight) {
//		int ww = (int) ( (float) uptime / ( (float) warmup / (float) weight ) );
		/**
		 * 原算法本机测试，一次耗时约14ns
		 * 新算法，一次耗时约96ns，为原算法时间复杂度的7倍左右。主要是Math.pow耗时
		 * 由于仅发布期间预热需要使用该方法，暂时直接替换
		 *
		 * 后期优化方向：
		 * 	-> 单独起线程计算本机预热期间的权重，从主流程中抽出
		 */
		// 预热权重从"一次正比例函数"修改为"指数函数"
		// 指数函数公式可查看：http://www.fooplot.com/?lang=zh_hans#W3sidHlwZSI6MCwiZXEiOiIxMDBeKHgvMzAwKSIsImNvbG9yIjoiIzAwMDAwMCJ9LHsidHlwZSI6MTAwMCwid2luZG93IjpbIi0xIiwiMzAwIiwiLTEiLCIxMjAiXX1d
		int ww = (int) Math.pow(weight, (double) uptime / warmup);

		if (ww < 1) {
			return 1;
		}
		return Math.min(ww, weight);
	}
}
