package cn.com.duibaboot.ext.autoconfigure.graceclose;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import cn.com.duibaboot.ext.autoconfigure.core.EarlyClose;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.devtools.classpath.ClassPathChangedEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.util.concurrent.TimeUnit;

@Configuration
public class GracefulCloseAutoConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(GracefulCloseAutoConfiguration.class);

    @Autowired
    private ApplicationContext applicationContext;
    //优雅关闭的等待时间，默认6秒
    @Value("${duiba.graceclose.wait.seconds:6}")
    private int graceCloseWaitSeconds;

    @Configuration
    @ConditionalOnClass(ClassPathChangedEvent.class)
    static class GracefulCloseDevToolsConfiguration{
        /**
         * 监听spring-boot-devtools的请求热重启事件
         * @param event
         */
        @EventListener
        @Order(Ordered.HIGHEST_PRECEDENCE)
        public void onClassPathChanged(ClassPathChangedEvent event) {
            if(event.isRestartRequired()){
                GracefulCloseLifeCycle.isDevToolRestarting = true;//NOSONAR
            }
        }
    }

    /**
     * 监听spring-boot的ContextRefreshedEvent
     * @param event
     */
    @EventListener
    public void onClassPathChanged(ContextRefreshedEvent event) {
        GracefulCloseLifeCycle.isDevToolRestarting = false;//NOSONAR
    }

    @Bean
    public GracefulCloseLifeCycle gracefulCloseLifeCycle(){

        //等待时间设置为6秒，让SLB或者nginx健康检查有足够的时间来调用/monitor/check接口发现服务不可用，从而关闭对本机的流量.(SLB/nginx一般配置每隔2秒检查一次/monitor/check)
        //需要配合运维的脚本生效，运维kill pid后需要等待16秒后再强杀应用。
        int sleepSeconds = graceCloseWaitSeconds;

        //当测试类上注解了@SpringBootTest时，spring会自动加上属性:org.springframework.boot.test.context.SpringBootTestContextBootstrapper=true
        String isInUnitTest = applicationContext.getEnvironment().getProperty("org.springframework.boot.test.context.SpringBootTestContextBootstrapper");
        //判断当前是否在执行单元测试,执行单测为了加快速度，不sleep
        if("true".equals(isInUnitTest)){
            sleepSeconds = 0;
        }

        return new GracefulCloseLifeCycle(sleepSeconds);
    }

    /**
     * 这个类的stop方法会被spring自动调用，spring会在取消注册eureka/consul后进入本类的stop方法从而sleep 6秒，让客户端(nginx or ribbon)有足够的时间检测到本服务已经下线从而实现优雅停机
     * （客户端会每隔3秒(ribbon是3秒，nginx是2秒)发送一次心跳，如果/monitor/check接口已经下线则移除对该服务器的访问）
     */
    public static class GracefulCloseLifeCycle extends EarlyClose{

        private final int sleepSeconds;

        /**
         * spring-boot-devtools是否正在重启
         */
        protected static volatile boolean isDevToolRestarting = false;

        public GracefulCloseLifeCycle(int sleepSeconds){
            this.sleepSeconds = sleepSeconds;
        }

        @Override
        public void stop() {

            if(!logger.isInfoEnabled()) {
                LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
                loggerContext.getLogger(GracefulCloseAutoConfiguration.class).setLevel(Level.INFO);
            }
            if(isDevToolRestarting){
                logger.info("detect devtools is restarting, close immediately");
                return;
            }
            //在调用到这里之前，GracefulCloseRunListener已经把/monitor/check接口置为返回FAIL
            logger.info("sleep {} seconds to wait nginx/ribbon check", sleepSeconds);
            try {
                TimeUnit.SECONDS.sleep(sleepSeconds);
            } catch (InterruptedException e) {
                //Ignore
                Thread.currentThread().interrupt();
            }

        }

        /**
         * Eureka/Consul的LifeCycle的phase是0，这里设置为-1
         * 1.让本类的stop会在取消注册之后被调用
         * 2.让本类晚于DubboGracefulCloseLifeCycle(主要负责dubbo优雅停机)执行
         *
         * @return
         */
        @Override
        public int getPhase() {
            return -1;
        }
    }
}
