package cn.com.duibaboot.ext.autoconfigure.perftest.dubbo;

import cn.com.duiba.boot.perftest.PerfTestConstant;
import cn.com.duibaboot.ext.autoconfigure.perftest.core.UnsupportedAppsHolder;
import cn.com.duiba.boot.perftest.PerfTestUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.rpc.Invocation;
import org.apache.dubbo.rpc.Invoker;
import org.apache.dubbo.rpc.RpcException;
import org.apache.dubbo.rpc.cluster.router.AbstractRouter;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 压测的dubbo服务客户端路由器，过滤出压测机器进行调用
 * Created by guoyanfei .
 * 2022/3/8 .
 */
public class PerfTestDubboRouter extends AbstractRouter {

    public PerfTestDubboRouter(URL url) {
        this.setUrl(url);
    }

    @Override
    public <T> List<Invoker<T>> route(List<Invoker<T>> invokers, URL url, Invocation invocation) throws RpcException {
        if (CollectionUtils.isEmpty(invokers)) {
            return invokers;
        }

        // 当前机器环境变量中的场景id
        String currentSceneId = PerfTestUtils.getSceneId();
        String targetAppName = null;

        List<Invoker<T>> testList = new ArrayList<>(invokers.size());
        List<Invoker<T>> prodList = new ArrayList<>(invokers.size());
        for (Invoker<T> invoker : invokers) {
            URL dubboUrl = invoker.getUrl();
            String targetSceneId = dubboUrl.getParameter(PerfTestConstant.PERF_TEST_SCENE_ID_DUBBO_KEY);

            if (StringUtils.isBlank(targetAppName)) {
                targetAppName = dubboUrl.getParameter(CommonConstants.REMOTE_APPLICATION_KEY);
            }

            if (sceneIdEquals(currentSceneId, targetSceneId)) { // 本机器和目标服务器有相同的场景id
                testList.add(invoker);
            }

            if (StringUtils.isBlank(targetSceneId)) {   // 生产的机器
                prodList.add(invoker);
            }
        }

        // 独立容器list不为空
        if (CollectionUtils.isNotEmpty(testList)) {
            return testList;
        }

        // 当前机器是压测机器，但是找不到相同场景id的压测机器，并且目标应用不在不支持压测的应用列表中，那么直接把所有机器都过滤掉，让他外层找不到服务报错
        if (StringUtils.isNotBlank(currentSceneId) && !UnsupportedAppsHolder.isUnsupportedApp(targetAppName)) {
            return Collections.emptyList();
        }

        // 其他所有情况，都走生产
        return CollectionUtils.isNotEmpty(prodList) ? prodList : invokers;
    }

    /**
     * 判断当前服务器环境变量的场景id和服务器注册在eureka的场景id是否相同
     * 任何一个为null，都认为不相同
     *
     * @param currentSceneId
     * @param targetSceneId
     * @return
     */
    private boolean sceneIdEquals(String currentSceneId, String targetSceneId) {
        if (currentSceneId == null || targetSceneId == null) {
            return false;
        }
        return currentSceneId.equals(targetSceneId);
    }
}
