/*
 * Copyright 2012-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.duibaboot.ext.autoconfigure.data.redis;

import cn.com.duiba.wolf.cache.Hessian2SerializationRedisSerializer;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.EnableAutoConfiguration;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.data.redis.connection.*;
import org.springframework.data.redis.connection.jedis.JedisConnection;
import org.springframework.data.redis.core.RedisOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;

import java.net.UnknownHostException;

/**
 * {@link EnableAutoConfiguration Auto-configuration} for Spring Data's Redis support.
 * Redis自动配置
 * 注意：如果duiba.redis.host没有配置，则spring自带的RedisAutoConfiguration会生效
 *
 * @author hwq
 */
@Configuration
@AutoConfigureBefore(org.springframework.boot.autoconfigure.data.redis.RedisAutoConfiguration.class)
@ConditionalOnClass({ JedisConnection.class, RedisOperations.class })
@EnableConfigurationProperties(RedisProperties.class)
@ConditionalOnProperty(name="duiba.redis.host", matchIfMissing = false)
@Import({LettuceConnectionConfiguration.class, JedisConnectionConfiguration.class})
public class RedisAutoConfiguration {

	/**
	 * Standard Redis configuration.
	 */
	@Configuration
	@ConditionalOnClass({ JedisConnection.class, RedisOperations.class })
	@ConditionalOnProperty(name="duiba.redis.host", matchIfMissing = false)
	protected static class RedisConfiguration {

		@Bean(name="redisTemplate")
		@ConditionalOnMissingBean(name = "redisTemplate")
		public RedisTemplate<Object, Object> redisTemplate(RedisConnectionFactory redisConnectionFactory)
						throws UnknownHostException {
			RedisTemplate<Object, Object> template = new RedisTemplate<>();
			template.setConnectionFactory(redisConnectionFactory);
			//使用hession2序列化方式，提高性能，减少占用redis的空间
			template.setDefaultSerializer(new Hessian2SerializationRedisSerializer());
			return template;
		}

		@Bean(name = "stringRedisTemplate")
		@ConditionalOnMissingBean(name = "stringRedisTemplate")
		public StringRedisTemplate stringRedisTemplate(RedisConnectionFactory redisConnectionFactory)
						throws UnknownHostException {
			StringRedisTemplate template = new StringRedisTemplate();
			template.setConnectionFactory(redisConnectionFactory);
			return template;
		}

		@Bean
		public Hessian2RedisDeserializeMvcEndpoint hessian2RedisDeserializeMvcEndpoint() {
			return new Hessian2RedisDeserializeMvcEndpoint();
		}

	}

}
