package cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.RocketMqProducerFlowReplaySpan;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.core.annotation.Order;

import java.lang.reflect.Method;

/**
 * 录制RocketMq的方法
 * Created by guoyanfei .
 * 2019-07-12 .
 */
@Slf4j
@Aspect
@Order  // 默认最低优先级，这个拦截需要在最后，因为其他的拦截可能会在方法参数中加kv，如果再前面执行会导致回放参数对比不一致
public class RecordRocketMqProducerPlugin {

    @Around("execution(* org.apache.rocketmq.client.producer.DefaultMQProducer+.*(..))")
    public Object rocketMqProducerJoinPoint(ProceedingJoinPoint joinPoint) throws Throwable {
        MethodSignature signature = (MethodSignature) joinPoint.getSignature();
        Method method = signature.getMethod();

        if (!FlowReplayTrace.isTraced() || !"send".equals(method.getName())) {
            return joinPoint.proceed();
        }
        // 本次调用已经被标记为需要忽略，不录制
        if (IgnoreSubInvokesContext.isMarked()) {
            return joinPoint.proceed();
        }

        Object[] parameterValues = joinPoint.getArgs();
        Object obj = joinPoint.getTarget();
        // 标记本次调用的子调用不需要录制
        IgnoreSubInvokesContext.instMark(obj, method.getName(), parameterValues);

        RocketMqProducerFlowReplaySpan span = null;
        try {
            // 此处先录下参数，在方法执行之后，再录下返回值，与其他的录制方式不同
            // 因为，在send方法中，参数中会被设置一些额外的内容进去，在方法执行完成之后录制，会把额外内容录制进去
            // 但是在回放的时候不会执行send方法，所以参数中不会有额外的内容，这样就会产生对比不一致而回放失败的情况
            span = RocketMqProducerFlowReplaySpan.createSpan(signature, parameterValues);
            span.setTraceId(FlowReplayTrace.getCurrentTraceId());
            FlowReplayTrace.addSubSpan(span);
        } catch (Throwable t) {
            log.error("RocketMq录制_录制异常", t);
        }

        Object ret;
        try {
            ret = joinPoint.proceed();
        } catch (Throwable t) {
            // 如果捕捉到异常，并且正在录制中，那么不录这个用例
            FlowReplayTrace.remove();
            IgnoreSubInvokesContext.unmark();
            throw t;
        }

        try {
            if (span != null) {
                span.setReturnValue(ret);
            }
        } catch (Throwable t) {
            log.error("RocketMq录制_录制异常", t);
        } finally {
            IgnoreSubInvokesContext.unmark();
        }
        
        return ret;
    }
}
