package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.*;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.RedisFlowReplaySpan;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.SpanType;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.List;

/**
 * 流量回放的redis方法拦截器
 * 把当前请求的方法名、参数列表，参数值列表和录制的trace中的进行对比，如果都相同，那么mock本次调用的返回值为trace中的返回值
 */
@Slf4j
public class ReplayJedisMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 回放的时候，直接忽略的方法
     */
    private static final List<String> REPLAY_FAST_MOCK_METHODS = Arrays.asList("expire",
                                                                               "expireAt",
                                                                               "pexpire",
                                                                               "pexpireAt");

    private static final List<String> SETEX_METHODS = Arrays.asList("setex",
                                                                    "psetex");

    /**
     * 可以回放
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        // 当前不是回放的环境 或者 当前请求不是回放请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }

        if (FlowReplayConstants.REDIS_AOP_EXCLUDE_METHOD.contains(method.getName())) {
            return false;
        }

        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) {   // NOSONAR
        if (!canReplay(method, allArguments)) {
            return;
        }
        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回放的span，用于回放详情的构建
        RedisFlowReplaySpan replayDetailSpan = createReplayDetailSpan(method, allArguments, argumentsTypes, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.REDIS != span.getSpanType()) {
            String expert = SpanType.REDIS.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        RedisFlowReplaySpan redisSpan = (RedisFlowReplaySpan) span;

        if (!FlowReplayUtils.isMethodEqual(redisSpan.getMethodName(), method)) {
            String expert = redisSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_102, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (REPLAY_FAST_MOCK_METHODS.contains(method.getName())) {
            try {
                Object mockRet = redisSpan.getReturnValue();
                // 参数都相同的情况下，mock之前把mock的返回值回填到构建回放详情用的span中
                replayDetailSpan.setReturnValue(mockRet);

                result.defineReturnValue(mockRet);
                return;
            } catch (KryoException e) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_104, e);
                throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
            }
        }

        if (!FlowReplayUtils.isArgumentsTypesEqual(redisSpan.getParameterTypes(), argumentsTypes)) {
            String expert = FlowReplayUtils.stringArrayToString(redisSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(argumentsTypes);
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_100, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            Object[] recordParameterValues = redisSpan.getParameterValues();
            Object[] replayParameterValues = allArguments;
            if (SETEX_METHODS.contains(method.getName())) {
                recordParameterValues = new Object[]{recordParameterValues[0], recordParameterValues[2]};
                replayParameterValues = new Object[]{replayParameterValues[0], replayParameterValues[2]};
            }
            if (!FlowReplayUtils.isArgumentsEqual(recordParameterValues, replayParameterValues)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_101);
                throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
            }
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_103, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            Object mockRet = redisSpan.getReturnValue();
            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回放详情用的span中
            replayDetailSpan.setReturnValue(mockRet);

            result.defineReturnValue(mockRet);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_104, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    private RedisFlowReplaySpan createReplayDetailSpan(Method method, Object[] allArguments, Class<?>[] argumentsTypes, FlowReplaySpan span) {
        RedisFlowReplaySpan replayDetailSpan = RedisFlowReplaySpan.createSpan(method, allArguments, argumentsTypes, null);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) {
        return ret;
    }

    @Override
    public void handleMethodException(Object objInst, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        //do nothing
    }

}
