package cn.com.duibaboot.ext.autoconfigure.batch;

import cn.com.duiba.boot.perftest.InternalPerfTestContext;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import org.springframework.util.ConcurrentReferenceHashMap;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 批处理请求
 * 将每个请求的参数，封装成list参数，调用指定的批量接口
 *
 * @author houwen
 */
public class BatchHandler {

    private static final Map<String, Method> findMethodCache = new ConcurrentReferenceHashMap<String, Method>(64);

    /**
     * 将请求参数合并，调用批量方法
     *
     * @param reqs
     * @throws Exception
     */
    public void doBatch(String bucketId, List<ReqContext> reqs) throws Exception {
        Transaction t = Cat.newTransaction("MergeRequest", bucketId);
        try {
            ReqContext one = reqs.get(0);
            // 来自压测请求的桶，把当前线程打上压测标记
            if (one.isCurrentInPerfTestMode()) {
                InternalPerfTestContext.markAsPerfTest(one.getCurrentPerfTestSceneId(), one.isTestCluster());
            }
            Method method = getMethod(one);
            Object[] args = getArgs(one.getParams().length, reqs);
            Object obj = method.invoke(one.getTarget(), args);
            //如果方法是 void 返回类型，直接返回
            if (Void.TYPE == method.getReturnType()) {
                return;
            }
            if (obj != null) {
                List<Object> ret = (List) obj;
                // 输入输出数量不同时抛异常 （这里需要批量接口内部保证数据兼容性）
                if (ret.size() != reqs.size()) {
                    throw new RuntimeException("input output size not equal"); // NOSONAR
                }
                // 将返回结果拆分，返回这一批请求的每个结果
                for (int i = 0; i < reqs.size(); i++) {
                    reqs.get(i).setResult(ret.get(i));
                }
            }
            t.setStatus(Message.SUCCESS);
        } catch (Exception e) {
            t.setStatus(e);
            throw e;
        } finally {
            InternalPerfTestContext.markAsNormal();
            t.complete();
        }
    }

    /**
     * 获取要执行的方法
     *
     * @return
     * @throws NoSuchMethodException
     */
    private Method getMethod(ReqContext req) throws NoSuchMethodException {
        Method cache = findMethodCache.get(req.getFullMethodName());
        if (cache != null) {
            return cache;
        }
        Method[] methods = req.getTarget().getClass().getMethods();
        for (Method m : methods) {
            if (m.getName().equals(req.getAnnotation().method())) {
                findMethodCache.put(req.getFullMethodName(), m);
                return m;
            }
        }
        throw new NoSuchMethodException(req.getAnnotation().method());
    }

    /**
     * 每一个请求参数，合并成 List 参数
     *
     * @param num
     * @param reqs
     * @return
     */
    private Object[] getArgs(int num, List<ReqContext> reqs) {
        Object[] args = new Object[num];
        for (int i = 0; i < num; i++) {
            List<Object> param = new ArrayList<>(reqs.size());
            for (ReqContext req : reqs) {
                param.add(req.getParams()[i]);
            }
            args[i] = param;
        }
        return args;
    }

}
