package cn.com.duibaboot.ext.autoconfigure.cat;

import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.dianping.cat.CatConstants;
import com.dianping.cat.configuration.client.entity.Server;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.DefaultMessageManager;
import com.dianping.cat.message.internal.DefaultTransaction;
import com.dianping.cat.util.Joiners;
import org.springframework.web.servlet.HandlerInterceptor;
import org.springframework.web.servlet.HandlerMapping;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.DispatcherType;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

/**
 * for servlet
 * Created by gyf .
 * 2017/11/30 .
 */
public class CatHandlerInterceptor implements HandlerInterceptor {

    public static final char SPLIT = '/';

    private static List<Handler> mHandlers = new ArrayList<>();

    static {
        mHandlers.add(CatHandlerInterceptor.CatHandler.ENVIRONMENT);
        mHandlers.add(CatHandlerInterceptor.CatHandler.LOG_CLIENT_PAYLOAD);
        mHandlers.add(CatHandlerInterceptor.CatHandler.ID_SETUP);
    }

    protected interface Handler {

        void handle(CatHandlerInterceptor.Context ctx) throws IOException, ServletException;
    }

    protected static class Context {

        private List<Handler> mHandlers;

        private int mIndex;

        private HttpServletRequest mRequest;

        private HttpServletResponse mResponse;

        private boolean mTop;

        private String mType;

        public Context(HttpServletRequest request, HttpServletResponse response, List<Handler> handlers) {
            mRequest = request;
            mResponse = response;
            mHandlers = handlers;
        }

        public HttpServletRequest getRequest() {
            return mRequest;
        }

        public HttpServletResponse getResponse() {
            return mResponse;
        }

        public String getType() {
            return mType;
        }

        public void handle() throws IOException, ServletException {
            if (mIndex < mHandlers.size()) {
                CatHandlerInterceptor.Handler handler = mHandlers.get(mIndex++);

                handler.handle(this);
            }
        }

        public boolean isTop() {
            return mTop;
        }

        public void setTop(boolean top) {
            mTop = top;
        }

        public void setType(String type) {
            mType = type;
        }
    }

    private enum CatHandler implements CatHandlerInterceptor.Handler {
        ENVIRONMENT {
            @Override
            public void handle(CatHandlerInterceptor.Context ctx) throws IOException, ServletException {
                HttpServletRequest req = ctx.getRequest();
                Message message = Cat.getManager().getThreadLocalMessageTree().getMessage();
                boolean top = message == null;

                ctx.setTop(top);

                if (top) {
                    ctx.setType(CatConstants.TYPE_URL);

                    setTraceMode(req);
                } else {
                    ctx.setType(CatConstants.TYPE_URL_FORWARD);
                }

                ctx.handle();
            }

            protected void setTraceMode(HttpServletRequest req) {
                String traceMode = "X-CAT-TRACE-MODE";
                String headMode = req.getHeader(traceMode);

                if ("true".equals(headMode)) {
                    Cat.getManager().setTraceMode(true);
                }
            }
        },

        ID_SETUP {
            private String mServers;

            private String getCatServer() {
                if (mServers == null) {
                    DefaultMessageManager manager = (DefaultMessageManager) Cat.getManager();
                    List<Server> servers = manager.getConfigService().getServers();

                    mServers = Joiners.by(',').join(servers, server -> {
                        if (server == null) {
                            return "";
                        }

                        String ip = server.getIp();
                        Integer httpPort = server.getHttpPort();

                        return ip + ":" + httpPort;
                    });
                }

                return mServers;
            }

            @Override
            public void handle(CatHandlerInterceptor.Context ctx) throws IOException, ServletException {
                boolean isTraceMode = Cat.getManager().isTraceMode();
                HttpServletResponse res = ctx.getResponse();

                if (isTraceMode) {
                    String id = Cat.getCurrentMessageId();

                    res.setHeader("X-CAT-ROOT-ID", id);
                    res.setHeader("X-CAT-SERVER", getCatServer());
                }

                ctx.handle();
            }
        },

        LOG_CLIENT_PAYLOAD {
            @Override
            public void handle(CatHandlerInterceptor.Context ctx) throws IOException, ServletException {
                HttpServletRequest req = ctx.getRequest();
                String type = ctx.getType();

                if (ctx.isTop()) {
                    logRequestClientInfo(req, type);
                    logRequestPayload(req, type);
                } else {
                    logRequestPayload(req, type);
                }

                ctx.handle();
            }

            protected void logRequestClientInfo(HttpServletRequest req, String type) {
                StringBuilder sb = new StringBuilder(1024);
                String ip = "";
                String ipForwarded = req.getHeader("x-forwarded-for");

                if (ipForwarded == null) {
                    ip = req.getRemoteAddr();
                } else {
                    ip = ipForwarded;
                }

                sb.append("IPS=").append(ip);
                sb.append("&VirtualIP=").append(req.getRemoteAddr());
                sb.append("&Server=").append(req.getServerName());
                sb.append("&Referer=").append(req.getHeader("referer"));
                sb.append("&Agent=").append(req.getHeader("user-agent"));

                Cat.logEvent(type, type + ".Server", Message.SUCCESS, sb.toString());
            }

            protected void logRequestPayload(HttpServletRequest req, String type) {
                StringBuilder sb = new StringBuilder(256);

                sb.append(req.getScheme().toUpperCase()).append('/');
                sb.append(req.getMethod()).append(' ').append(req.getRequestURI());

                String qs = req.getQueryString();

                if (qs != null) {
                    sb.append('?').append(qs);
                }

                Cat.logEvent(type, type + ".Method", Message.SUCCESS, sb.toString());
            }
        },

    }

    private void customizeStatus(Transaction t, HttpServletRequest req) {
        Object catStatus = req.getAttribute(CatConstants.CAT_STATE);

        if (catStatus != null) {
            t.setStatus(catStatus.toString());
        } else {
            t.setStatus(Message.SUCCESS);
        }
    }

    private void customizeUri(Transaction t, HttpServletRequest req) {
        if (t instanceof DefaultTransaction) {
            Object catPageType = req.getAttribute(CatConstants.CAT_PAGE_TYPE);

            if (catPageType instanceof String) {
                ((DefaultTransaction) t).setType(catPageType.toString());
            }

            Object catPageUri = req.getAttribute(CatConstants.CAT_PAGE_URI);

            if (catPageUri instanceof String) {
                ((DefaultTransaction) t).setName(catPageUri.toString());
            }
        }
    }

    @Override
    public boolean preHandle(HttpServletRequest request, HttpServletResponse response, Object handler) throws Exception {
        if(!CatUtils.isCatEnabled()){
            return true;
        }
        if ((DispatcherType.FORWARD != request.getDispatcherType()) && (DispatcherType.REQUEST != request.getDispatcherType())) {
            return true;
        }

        CatHandlerInterceptor.Context ctx = new CatHandlerInterceptor.Context(request, response, mHandlers);
        ctx.handle();

        String uri = (String) request.getAttribute(HandlerMapping.BEST_MATCHING_PATTERN_ATTRIBUTE);
//        if(uri.endsWith("**")){// spring可以处理@RequestMapping("/zipkin/api/v1/**")这样的请求，如果是这种则获取具体的url
//            uri = (String)request.getAttribute(HandlerMapping.PATH_WITHIN_HANDLER_MAPPING_ATTRIBUTE);
//        }
        Cat.newTransaction(ctx.getType(), uri);
        return true;
    }

    @Override
    public void postHandle(HttpServletRequest request, HttpServletResponse response, Object handler, ModelAndView modelAndView) throws Exception {
        //do nothing
    }

    @Override
    public void afterCompletion(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) throws Exception {
        if ((DispatcherType.FORWARD != request.getDispatcherType()) && (DispatcherType.REQUEST != request.getDispatcherType())) {
            return;
        }
        Transaction t = Cat.getManager().getPeekTransaction();
        if (t == null) {
            return;
        }
        try {
            if (ex == null) {
                customizeStatus(t, request);
            } else {
                t.setStatus(ex);
                Cat.logError(ex);
            }
        } finally {
            customizeUri(t, request);
            t.complete();
        }
    }
}
