package cn.com.duibaboot.ext.autoconfigure.core.utils;

import cn.com.duiba.boot.utils.NetUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.io.buffer.DataBuffer;
import org.springframework.core.io.buffer.NettyDataBufferFactory;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.http.server.reactive.ServerHttpResponse;
import reactor.core.publisher.Mono;

import java.nio.ByteBuffer;
import java.nio.charset.Charset;

/**
 * http请求工具类，reactive版
 */
public class ReactiveHttpRequestUtils {
    private ReactiveHttpRequestUtils(){}

    /**
     * 获取用户的ip地址
     * @param request
     * @return
     */
    public static String getIpAddr(ServerHttpRequest request) {
        String ip = request.getHeaders().getFirst("x-forwarded-for");
        if (ip != null && ip.trim().length() > 0) {
            String[] ips = ip.trim().split(",");
            int size = ips.length;
            if (size > 0) {
                ip = ips[0].trim();
            }
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("X-Real-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("WL-Proxy-Client-IP");
        }
        if (isInvalidIp(ip)) {
            ip = request.getHeaders().getFirst("Cdn-Src-Ip");
        }
        if (isInvalidIp(ip)) {
            ip = request.getRemoteAddress().getAddress().getHostAddress();
        }
        if (ip != null && ip.startsWith("0:0:0:0")) {
            ip = "127.0.0.1";//NOSONAR
        }
        return ip;
    }

    private static boolean isInvalidIp(String ip){
        return ip == null || ip.length() == 0 || HttpRequestUtils.UNKNOWN.equalsIgnoreCase(ip);
    }

    /**
     * 访问的来源ip必须是127.0.0.1， 并且访问的url也必须是127.0.0.1
     * @param request
     * @return
     */
    public static boolean isLocalRequest(ServerHttpRequest request){
        String ip = getIpAddr(request);
        if (!ip.equals("127.0.0.1") && !ip.equals("0:0:0:0")) {//NOSONAR
            return false;
        }

        //ip有可能是伪造的，故需要结合host一起判断
        boolean isUrlFromLocal = false;
        String host = request.getURI().getHost();
        if(host.equals("localhost") || host.equals("127.0.0.1")) {//NOSONAR
            isUrlFromLocal = true;
        }

        return isUrlFromLocal;
    }

    /**
     * 访问的来源ip必须在内网，并且没有经过代理
     * @param request
     * @return
     */
    public static boolean isLanRequest(ServerHttpRequest request){
        if(StringUtils.isNotBlank(request.getHeaders().getFirst("x-forwarded-for"))){
            // 有x-forwarded-for字段表示来自代理。
            return false;
        }
        String ip = getIpAddr(request);
        boolean isLanIp = false;
        if (ip.equals("127.0.0.1") || ip.equals("0:0:0:0") || NetUtils.isLanIp(ip)) {//NOSONAR
            isLanIp = true;
        }
        if(!isLanIp){
            return false;
        }

        //ip有可能是伪造的，故需要结合host一起判断
        boolean isUrlFromLan = false;
        String host = request.getURI().getHost();
        if (host.equals("127.0.0.1") || host.equals("localhost") || NetUtils.isLanIp(host)) {//NOSONAR
            isUrlFromLan = true;
        }

        return isUrlFromLan;
    }

    public static Mono<Void> write(ServerHttpResponse response, String message){
        ByteBuffer byteBuffer = Charset.forName("UTF-8").encode(message);
        NettyDataBufferFactory factory = (NettyDataBufferFactory) response.bufferFactory();
        DataBuffer buffer = factory.wrap(byteBuffer);
        response.getHeaders().setContentLength(byteBuffer.remaining());
        return response.writeWith(Mono.just(buffer));
    }

}
